package com.calpano.common.client;

import java.util.HashSet;
import java.util.Set;

import org.xydra.annotations.RunsInGWT;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.util.CommonAppState;

/**
 * Models the client side global application state.
 *
 * Singleton.
 *
 * @author xamde
 */
@RunsInGWT(true)
public class AppState extends CommonAppState {

	private static final Logger log = LoggerFactory.getLogger(AppState.class);

	/**
	 * Listen to changes in the AppState
	 */
	public static interface AppStateListener {
		void onAppStateChange();
	}

	/**
	 * Start is always {@link Stage#BeforeScript}. On load, we switch to
	 * {@link Stage#Starting}.
	 *
	 * @author xamde
	 */
	public static enum Stage {

		/** GWT has not started yet, this is loaded plain from the DOM */
		BeforeScript(true),

		/**
		 * GWT started and our code started to run. We let the GWT app try to
		 * init.
		 *
		 * <b>When we are halfway done rendering anything the first user input
		 * might come.</b>
		 */
		Starting(true),

		/** We should not render the app on this URL. */
		DontShowApp(false),

		/** We'Re stuck. We need to reboot. */
		Bluescreen(false);

		private final boolean busy;

		Stage(final boolean busy) {
			this.busy = busy;
		}

		public boolean isBusy() {
			return this.busy;
		}

		static {
			BeforeScript.setNextStages(Starting, Bluescreen, DontShowApp);
			Starting.setNextStages(Bluescreen);
			// terminal states
			Bluescreen.setNextStages();
			DontShowApp.setNextStages();
		}

		Set<Stage> nextStages = new HashSet<AppState.Stage>();

		public Set<Stage> getNextStages() {
			return this.nextStages;
		}

		public boolean isDeterministic() {
			return this.nextStages.size() < 2;
		}

		public boolean isFinal() {
			return this.nextStages.size() == 0;
		}

		public void setNextStages(final Stage... next) {
			for (final Stage s : next) {
				this.nextStages.add(s);
			}
		}

	}

	/** Was I ever able to contact the server since the script booted? */
	private static Process successfulAjax = Process.Never;

	private static Process successfulRender = Process.Never;

	/** Did the server say yes to us? */
	private static boolean authenticatedOnServer_ = false;

	private static Stage currentStage = Stage.Starting;

	private static Set<AppStateListener> listeners = new HashSet<AppState.AppStateListener>();

	private static int openAjaxCalls = 0;

	private static int openRenderCalls = 0;

	public static void addListener(final AppStateListener listener) {
		listeners.add(listener);
	}

	public static Stage getStage() {
		return currentStage;
	}

	/**
	 * @return true if we can get (got at least one) AJAX response (i.e. no
	 *         server error). That means we're online since booting the GWT.
	 */
	public static boolean isOnline() {
		return successfulAjax == Process.DoneSuccess;
	}

	public static void printState(final String s) {
		log.info("===  " + s + "===");
		log.info("stage=" + currentStage);
		log.info("initialRender=" + successfulRender + "\n");
		log.info("initialAjax=" + successfulAjax + "\n");
		log.info("online=" + isOnline() + "\n");
		log.info("authenticatedOnServer=" + authenticatedOnServer_ + "\n");
		log.info("Open AJAX calls=" + openAjaxCalls + "\n");
		log.info("Open Render calls=" + openRenderCalls + "\n");

	}

	public static boolean isAuthenticatedOnServer() {
		return authenticatedOnServer_;
	}

	public static boolean isWaitingForAjax() {
		return openAjaxCalls > 0 || successfulAjax == Process.Started;
	}

	public static boolean isWaitingForRendering() {
		return openRenderCalls > 0 || successfulRender == Process.Started;
	}

	private static void onChange() {
		switch (currentStage) {
		case BeforeScript:
			throw new AssertionError("Cannot happen in script-part");
		case Starting:
			// fine
			break;
		case DontShowApp:
		case Bluescreen:
			// Terminal states
		}

	}

	public static void removeListener(final AppStateListener listener) {
		listeners.remove(listener);
	}

	public static void setIsAuthenticatedOnServer(final boolean authenticated) {
		authenticatedOnServer_ = authenticated;
		onChange();
	}

	public static void setStage(final Stage stage) {
		assert currentStage.getNextStages().contains(stage) || stage == currentStage;
		final boolean newStage = stage != currentStage;
		currentStage = stage;
		if (newStage) {
			for (final AppStateListener l : listeners) {
				l.onAppStateChange();
			}
		}
	}

	/**
	 * @return true if there was at least one successful render process after
	 *         GWT booted
	 */
	public static Process isSuccessfulRender() {
		return successfulRender;
	}

	/**
	 * @return true if there was at least one successful AJAX call to the server
	 *         after GWT booted, success means here no server error
	 */
	public static Process isSuccessfulAjax() {
		return successfulAjax;
	}

	public static void setStageSilently(final Stage stage) {
		currentStage = stage;
	}

	public static int getOpenAjaxCalls() {
		return openAjaxCalls;
	}

	public static int getOpenRenderCalls() {
		return openRenderCalls;
	}
}
