package com.calpano.common.client.commands;

import org.xydra.annotations.CanBeNull;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.XyAssert;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.RepeatingCommand;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;

/**
 * Provides ways to start repeating commands in a way so that they cannot
 * interfere with each other.
 *
 * @author xamde
 */
public class SynchronizedScheduler {

	/**
	 * A {@link ScheduledCommand} that can be executed via several other
	 * {@link ScheduledCommand} and is guaranteed to execute exclusively before
	 * any other {@link ISynchronizedCommand} is running.
	 *
	 * Implementations must call {@link #leaveCriticalSection()} in the
	 * {@link #execute()} method if no further commands are spawned.
	 *
	 * @author xamde
	 */
	private static abstract class AbstractSynchronizedCommand {

		protected final long id;

		public AbstractSynchronizedCommand() {
			this.id = idGen++;
		}

		public boolean canRun() {
			return isIdle() || myIdIsActive();
		}

		/**
		 * Harmless to call several times
		 */
		protected void enterCriticalSection() {
			activeId = this.id;
		}

		public boolean myIdIsActive() {
			return activeId == this.id;
		}

	}

	/**
	 * Helper callback to wait for a number of sub-commands to finish
	 *
	 * @author xamde
	 */
	public static class AggregateDoneCallback implements DoneCallback {
		private int calls = 0;
		private final DoneCallback combinedCallback;
		private final String debugName;
		private int expectedCalls;

		/**
		 * @param debugName
		 * @param expectedCalls
		 *            number of times this callback needs to be called before
		 *            combinedCallback gets notified once
		 * @param combinedCallback
		 *            gets notified once
		 */
		public AggregateDoneCallback(final String debugName, final int expectedCalls,
				final DoneCallback combinedCallback) {
			this.debugName = debugName;
			this.expectedCalls = expectedCalls;
			this.combinedCallback = combinedCallback;
		}

		@Override
		public void done() {
			this.calls++;
			log.debug((this.calls == this.expectedCalls ? "Completed" : "Did") + " " + this.calls
					+ "/" + this.expectedCalls + " of '" + this.debugName + "'");
			XyAssert.xyAssert(this.calls <= this.expectedCalls,
					"Received more calls than expected: %s instead of %s", this.calls,
					this.expectedCalls);
			if (this.calls == this.expectedCalls) {
				this.combinedCallback.done();
			}
		}

		public void expectOneMore() {
			this.expectedCalls++;
		}

	}

	private static class SynchronizedRepeatingCommand extends AbstractSynchronizedCommand implements
			RepeatingCommand {

		private final DoneCallback doneCallback;
		private final RepeatingCommand repeatingCommand;

		/**
		 * @param repeatingCommand
		 * @param doneCallback
		 *            @CanBeNull
		 */
		public SynchronizedRepeatingCommand(final RepeatingCommand repeatingCommand,
				@CanBeNull final DoneCallback doneCallback) {
			super();
			this.doneCallback = doneCallback;
			this.repeatingCommand = repeatingCommand;
		}

		@Override
		public boolean execute() {
			if (canRun()) {
				enterCriticalSection();
				final boolean repeat = this.repeatingCommand.execute();
				if (!repeat) {
					leaveCriticalSection();
					if (this.doneCallback != null) {
						this.doneCallback.done();
					}
				}
				return repeat;
			} else {
				// stop running as repeating command and reschedule for later
				Scheduler.get().scheduleIncremental(this);
				return false;
			}
		}

	}

	private static class SyncronizedScheduledCommand extends AbstractSynchronizedCommand implements
			ScheduledCommand {

		private final ScheduledCommand command;
		private final DoneCallback doneCallback;

		public SyncronizedScheduledCommand(final ScheduledCommand scheduledCommand,
				@CanBeNull final DoneCallback doneCallback) {
			super();
			this.command = scheduledCommand;
			this.doneCallback = doneCallback;
		}

		@Override
		public void execute() {
			// postpone your own execution
			if (canRun()) {
				enterCriticalSection();
				this.command.execute();
				leaveCriticalSection();
				if (this.doneCallback != null) {
					this.doneCallback.done();
				}
			} else {
				Scheduler.get().scheduleDeferred(this);
			}
		}

	}

	private static long activeId = 0;

	private static long idGen = 1;

	private static SynchronizedScheduler INSTANCE = new SynchronizedScheduler();

	private static final Logger log = LoggerFactory.getLogger(SynchronizedScheduler.class);

	/**
	 * @return the singleton instance of this scheduler
	 */
	public static SynchronizedScheduler get() {
		return INSTANCE;
	}

	/**
	 * @return true if no sSynchronized command is active
	 */
	public static boolean isIdle() {
		return activeId == 0;
	}

	private static void leaveCriticalSection() {
		activeId = 0;
	}

	/**
	 * Schedule a command that cannot run while any other ISynchronizedCommand
	 * is active.
	 *
	 * @param synchronizedCmd
	 *            to be scheduled
	 * @param doneCallback
	 *            @CanBeNull
	 */
	public void scheduleSynchronized(final ScheduledCommand synchronizedCmd, final DoneCallback doneCallback) {
		final SyncronizedScheduledCommand wrapper = new SyncronizedScheduledCommand(synchronizedCmd,
				doneCallback);
		Scheduler.get().scheduleDeferred(wrapper);
	}

	/**
	 * @param repeatingCommand
	 * @param doneCallback
	 *            @CanBeNull gets notified when the whole repeating command
	 *            inclusive all its repetitions are done.
	 *
	 *            Consider using a {@link AggregateDoneCallback} if you spawn
	 *            more {@link RepeatingCommand}s within the
	 *            {@link RepeatingCommand} to be scheduled here.
	 */
	public void scheduleSynchronized(final RepeatingCommand repeatingCommand, final DoneCallback doneCallback) {
		final SynchronizedRepeatingCommand wrapper = new SynchronizedRepeatingCommand(repeatingCommand,
				doneCallback);
		Scheduler.get().scheduleIncremental(wrapper);
	}

}
