package com.calpano.common.client.data;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.xydra.base.id.UUID;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.data.DataManager.AllEventsOccuredCallback;
import com.calpano.common.shared.data.DataEvent;
import com.calpano.common.shared.data.DataEvent.DataEventHandler;
import com.calpano.common.shared.data.DataEventCondition;
import com.google.web.bindery.event.shared.Event.Type;
import com.google.web.bindery.event.shared.EventBus;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * The {@link DataEventWatcher} waits for multiple {@link DataEvent}s to arrive
 * at the {@link EventBus} and to fire a final {@link AllEventsOccuredCallback}.
 *
 *
 * The watcher listens to a specified {@link EventBus} for {@link DataEvent}s
 * contained in the specified {@link DataEventCondition}s. When an event is
 * observed, it is checked against the given conditions. If it matches a
 * condition, that condition is removed. When all conditions are removed in this
 * way, the watcher fires the callback and is finished (and therefore does not
 * do anything anymore)
 *
 * @author alpha
 * @author xamde
 */
@SuppressWarnings("rawtypes")
class DataEventWatcher implements DataEventHandler {

	private static final Logger log = LoggerFactory.getLogger(DataEventWatcher.class);

	/**
	 * Initialize watcher, register at event bus and map event conditions to
	 * event types. Does not watch into already happened events.
	 *
	 * See {@link DataEventWatcher}
	 *
	 * @param eventBus
	 *            the {@link EventBus} to observe
	 * @param callback
	 *            callback to be invoked when all event conditions occurred
	 * @param eventConditions
	 *            the event conditions to watch for
	 */
	public static synchronized void watch(final EventBus eventBus, final AllEventsOccuredCallback callback,
			final DataEventCondition... eventConditions) {
		final DataEventWatcher dew = new DataEventWatcher(callback, eventConditions);
		dew.startWatching(eventBus);
	}

	/** rule action */
	private final AllEventsOccuredCallback callback;

	/** rule conditions */
	private final ArrayList<DataEventCondition> conditions;

	// TODO kill?
	private final String debugID = UUID.uuid(4);

	/** Map types to DataEventHandlers watching them */
	private final Map<Type<DataEventHandler<?>>, List<DataEventCondition>> eventTypeAwaitedConditionsMap;

	/** internal map of event listeners to get rid of no-longer-required ones */
	private final Map<Type<DataEventHandler<?>>, HandlerRegistration> eventTypeHandlerRegistrationMap;

	private DataEventWatcher(final AllEventsOccuredCallback callback,
			final DataEventCondition... eventConditions) {

		this.conditions = new ArrayList<DataEventCondition>(Arrays.asList(eventConditions));
		this.eventTypeAwaitedConditionsMap = new HashMap<Type<DataEventHandler<?>>, List<DataEventCondition>>();

		this.eventTypeHandlerRegistrationMap = new HashMap<Type<DataEventHandler<?>>, HandlerRegistration>();
		this.callback = callback;
	}

	/**
	 * Indicates whether all events occurred a.k.a. the watcher is done.
	 *
	 * @return true - When the watcher does not wait for any condition for any
	 *         type of event a.k.a. the watcher is done.
	 */
	private synchronized boolean allEventsOccured() {
		return this.eventTypeAwaitedConditionsMap.isEmpty();
	}

	@Override
	public synchronized void onData(final DataEvent event) {
		log.debug(this.debugID + " onData " + event);
		// Get the conditions for this type of event
		final List<DataEventCondition> eventConditionsForType = this.eventTypeAwaitedConditionsMap
				.get(event.getAssociatedType());

		// but if there are no conditions
		if (eventConditionsForType == null) {
			/*
			 * This watcher receives an event at time X of a type (that it
			 * previously registered on at the eventBus) and that it removes its
			 * registration from right after receiving it b/c the remaining
			 * conditions for this event type are matched. As the order in which
			 * the eventBus notifies listeners is undefined the watcher may
			 * receive additional events of the registered type event at time (X
			 * + DELTA) even though the watcher removed the registration at time
			 * X. If this happens, eventConditionsForType will be null, as the
			 * watcher is no longer interested in this type of events. Therefore
			 * it ignores those types of events.
			 */
			log.debug(this.debugID
					+ " eventConditionsForType is null, this watcher does not care about the event type anymore.");
			return; // ignore event
		}

		log.debug(this.debugID + " eventConditionsForType " + eventConditionsForType);

		/*
		 * Check the conditions for this type of event for matches, remove
		 * matching conditions
		 */
		final Iterator<DataEventCondition> it = eventConditionsForType.iterator();
		while (it.hasNext()) {
			final DataEventCondition condition = it.next();
			if (condition.matches(event)) {
				log.debug(this.debugID + " removing matching condition " + condition);
				it.remove();
			}
		}
		/*
		 * When there is no condition left for this type of event, we no longer
		 * need to observe events of this type
		 */
		if (eventConditionsForType.isEmpty()) {
			log.debug(this.debugID + " eventConditionsForType is empty now.");
			/*
			 * Clean up and remove the now empty list of conditions for this
			 * type of event from the map
			 */
			this.eventTypeAwaitedConditionsMap.remove(event.getAssociatedType());
			/*
			 * No longer observe this type of event: Remove the handler
			 * registration from the map and eventBus.
			 */
			final HandlerRegistration reg = this.eventTypeHandlerRegistrationMap.remove(event
					.getAssociatedType());
			reg.removeHandler();
		}
		/*
		 * When there is no condition for any type of event left, the watcher is
		 * done
		 */
		if (allEventsOccured()) {
			log.debug(this.debugID
					+ " eventTypeAwaitedConditionsMap is empty now. (all conditions met)");
			this.callback.onAllEventsOccured();
		}
	}

	/**
	 * Start watching for future events meeting the expected conditions
	 *
	 * @param eventBus
	 */
	private void startWatching(final EventBus eventBus) {
		for (final DataEventCondition eventCondition : this.conditions) {
			final Type<DataEventHandler<?>> type = eventCondition.getAssociatedType();
			// wait only for it, if we don't wait already for it
			if (!this.eventTypeAwaitedConditionsMap.containsKey(type)) {
				this.eventTypeAwaitedConditionsMap.put(type, new ArrayList<DataEventCondition>());
			}
			this.eventTypeAwaitedConditionsMap.get(type).add(eventCondition);
			if (!this.eventTypeHandlerRegistrationMap.containsKey(type)) {
				this.eventTypeHandlerRegistrationMap.put(type,
						eventBus.addHandler(eventCondition.getAssociatedType(), this));
			}
		}
	}

	/**
	 * Reset watcher, unregister from event bus and clear the maps from event
	 * type to registrations and event conditions.
	 */
	public synchronized void unwatch() {
		final Iterator<Type<DataEventHandler<?>>> it = this.eventTypeHandlerRegistrationMap.keySet()
				.iterator();
		while (it.hasNext()) {
			final Type<DataEventHandler<?>> type = it.next();
			final HandlerRegistration reg = this.eventTypeHandlerRegistrationMap.get(type);
			reg.removeHandler();
			it.remove();
		}
		this.eventTypeHandlerRegistrationMap.clear();
		this.eventTypeAwaitedConditionsMap.clear();
	}
}
