package com.calpano.common.client.storage;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.Setting;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.google.gwt.storage.client.Storage;

/**
 * Generic API for local storage as provided in HTML5
 *
 * @author xamde
 *
 * @param <T>
 */
public abstract class GenericStorage<T> {

	@Setting("Compile time flag to test absence")
	public static final boolean ENABLED = true;

	private static final Logger log = LoggerFactory.getLogger(GenericStorage.class);

	private static Storage storage = getLocalStorageIfSupported();

	/**
	 * Just does nothing if there is no local storage.
	 */
	public static void clearAllLocalStorage() {
		final Storage storage = getLocalStorageIfSupported();
		if (storage != null) {
			storage.clear();
			log.info("Local storage cleared");
		}
	}

	public static boolean isLocalStorageSupported() {
		return storage != null;
	}

	/**
	 * @return null if not supported
	 */
	public static Storage getLocalStorageIfSupported() {
		log.debug("Probing for localStorage support...");
		/* Work around a security DOM error in Firefox */
		try {
			Storage storage = Storage.getLocalStorageIfSupported();
			if (!ENABLED) {
				storage = null;
			}
			log.debug("Probing for localStorage support "
					+ (storage == null ? "negative" : "positive"));
			return storage;
		} catch (final Throwable t) {
			log.warn("Got exception when trying to detect localStorage", t);
		}
		log.debug("Probing for localStorage support negative");
		return null;
	}

	/**
	 * Just does nothing if localStorage not supported.
	 *
	 * @param key
	 * @param object
	 */
	public void store(final String key, final T object) {
		if (GenericStorage.storage == null) {
			log.info("LocalStorage is not supported by Client.");
		} else {
			final String data = serialize(object);
			try {
				/**
				 * work around this bug
				 * http://stackoverflow.com/questions/2603682
				 */
				GenericStorage.storage.removeItem(key);
				GenericStorage.storage.setItem(key, data);
			} catch (final Exception e) {
				log.warn("Exception in localStorage. Serialized attribute object<T> may be too large, length was "
						+ data.length());
				log.warn(e.toString());
			}
		}
	}

	/**
	 * @param key
	 * @return stored value in deserialised form or null of not found, null was
	 *         stored or local storage is not supported at all.
	 */
	public @CanBeNull T get(final String key) {
		if (GenericStorage.storage == null) {
			log.info("LocalStorage is not supported by client.");
			return null;
		}
		final String data = GenericStorage.storage.getItem(key);
		if (data == null) {
			log.info("Key " + key + " not found in localStorage.");
			return null;
		}
		// else
		final T object = deserialize(data);
		return object;
	}

	public void clear() {
		if (GenericStorage.storage != null) {
			GenericStorage.storage.clear();
		}
	}

	public void removeItem(final String key) {
		if (GenericStorage.storage != null) {
			GenericStorage.storage.removeItem(key);
		}
	}

	protected abstract String serialize(T object);

	protected abstract T deserialize(String data);

}
