package com.calpano.common.client.view.forms.impl;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.xydra.annotations.Feature;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.XyAssert;

import com.calpano.common.client.ClientApp;
import com.calpano.common.client.util.Callback;
import com.calpano.common.client.view.forms.IBelongsToHtml5Form;
import com.calpano.common.client.view.forms.IFormManagingWidget;
import com.calpano.common.client.view.forms.IFormSubmitEventHandler;
import com.calpano.common.client.view.forms.IHtml5EnabledInputWidget;
import com.calpano.common.client.view.forms.IViewValidator;
import com.calpano.common.client.view.forms.locking.ILockable;
import com.calpano.common.client.view.forms.locking.impl.LockUtils;
import com.calpano.common.client.view.forms.utils.FocusUtils;
import com.calpano.common.client.view.forms.validation.HasInvaliationHandlers;
import com.calpano.common.client.view.forms.validation.HasValidationHandlers;
import com.calpano.common.client.view.forms.validation.InvalidationEvent;
import com.calpano.common.client.view.forms.validation.InvalidationHandler;
import com.calpano.common.client.view.forms.validation.ValidationEvent;
import com.calpano.common.client.view.forms.validation.ValidationHandler;
import com.calpano.common.client.view.resources.CommonResourceBundle;
import com.calpano.common.shared.util.CommonAppState;
import com.calpano.common.shared.util.CommonAppState.Is;
import com.calpano.common.shared.validation.ValidationMessage;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HasWidgets;
import com.google.gwt.user.client.ui.Widget;
import com.google.web.bindery.event.shared.HandlerRegistration;

/**
 * Represents a grouping of input elements and buttons. Renders as a DIV with
 * class 'form'.
 *
 * OnLoad this form gathers automatically all DOM-children that are text inputs
 * and {@link Html5SubmitButton}.
 *
 * <h3>Listeners and Handler</h3> <b>Important:</b> At least one
 * {@link IFormSubmitEventHandler} must be registered and actually trigger the
 * submit in the {@link IFormSubmitEventHandler#onJustBeforeFormSubmit()}. It
 * should also handle the other events.
 *
 * <h3>Double-submits</h3> During a submit, the form is locked until it is
 * reset.
 *
 * * <h3>Validation</h3> A basic validation respecting 'required' and
 * 'type=email' is performed automatically. Custom validation can be used
 * instead by adding a {@link IViewValidator}.
 *
 *
 * <h3>Form life-cycle</h3> (1) Reset: Initial view: Showing no errors, even if
 * required fields are empty; not all fields/buttons are enabled. (2) Editing:
 * User leaves a field that is invalid (not just empty); Show error. (3)
 * InvalidSubmit: User tries to submit an invalid form (i.e. missing required);
 * Show the errors. (4) ValidSubmit: Lock form and wait.
 */
public class Html5FormPanel extends FlowPanel implements Callback,
/* so that the form can disable or enable the submit button */
HasValidationHandlers, HasInvaliationHandlers, ILockable {

	private static final Logger log = LoggerFactory.getLogger(Html5FormPanel.class);

	private final List<IFormSubmitEventHandler> formSubmitEventHandlers = new ArrayList<IFormSubmitEventHandler>();

	private IFormManagingWidget formSubmitter = null;

	/** true iff form is being submitted and waits for #reset */
	private boolean isSubmitting;

	/** true if form is valid, result of last validity change event */
	@Feature("validation")
	private CommonAppState.Is isValid = Is.Maybe;

	/** Can be null if no button is used */
	private Html5SubmitButton submitButton = null;

	/** List of input elements to use for validation */
	@Feature("validation")
	private final List<IHtml5EnabledInputWidget> textInputs = new ArrayList<IHtml5EnabledInputWidget>(1);

	private final List<HandlerRegistration> validationHandlerRegistrations = new ArrayList<HandlerRegistration>();

	public Html5FormPanel() {
		super();
		CommonResourceBundle.INSTANCE.css().ensureInjected();
	}

	/**
	 * The managing widget should register.
	 *
	 * TODO clarify role of this handler
	 *
	 * @see IFormSubmitEventHandler
	 * @param formEventHandler
	 *            gets informed about onJustBeforeSubmit, onReset and onFailed
	 */
	public void addFormSubmitEventHandler(final IFormSubmitEventHandler formEventHandler) {
		this.formSubmitEventHandlers.add(formEventHandler);
	}

	public void removeFormSubmitEventHandler(final IFormSubmitEventHandler formEventHandler) {
		this.formSubmitEventHandlers.remove(formEventHandler);
	}

	/**
	 * Computes the validation state of the form from the validation state of
	 * its child input elements
	 */
	@Feature("validation")
	private void computeValidation() {
		for (final IHtml5EnabledInputWidget h : this.textInputs) {
			final ValidationMessage vmsg = h.asHtml5TextInput().computeValidation();
			if (!vmsg.level.isValid()) {
				this.isValid = Is.No;
				return;
			}
		}
		this.isValid = Is.Yes;
	}

	private void findAllChildrenAndAddThem(final HasWidgets w) {
		int inputs = 0;
		int submits = 0;

		final InputValidationHandler inputValidationHandler = new InputValidationHandler();

		final Iterator<Widget> it = w.iterator();
		while (it.hasNext()) {
			final Widget child = it.next();
			if (child instanceof IBelongsToHtml5Form) {
				final IBelongsToHtml5Form html5Submit = (IBelongsToHtml5Form) child;
				html5Submit.setForm(this);
				if (child instanceof IHtml5EnabledInputWidget) {
					inputs++;
					final IHtml5EnabledInputWidget h5box = (IHtml5EnabledInputWidget) child;
					this.textInputs.add(h5box);
					/* listen to all validity change events from children */
					final HandlerRegistration reg = h5box.asHtml5TextInput().addValidationHandler(
							inputValidationHandler);
					// save in order to later unregister handler
					this.validationHandlerRegistrations.add(reg);
				} else if (child instanceof Html5SubmitButton) {
					submits++;
					assert this.submitButton == null : "you have added 2 submit buttons";
					this.submitButton = (Html5SubmitButton) child;
				}
			}
			// recurse
			if (child instanceof HasWidgets) {
				findAllChildrenAndAddThem((HasWidgets) child);
			}
		}
		log.trace("Form has " + inputs + " inputs and " + submits + " submits (should be just one");
	}

	@Feature("validation")
	private class InputValidationHandler implements ValidationHandler, InvalidationHandler {
		@Override
		public void onInvalid(final InvalidationEvent event) {
			/* if any child turns invalid, the form is invalid */
			Html5FormPanel.this.isValid = Is.No;
			if (Html5FormPanel.this.submitButton != null) {
				Html5FormPanel.this.submitButton.setEnabled(false);
			}
		}

		@Override
		public void onValid(final ValidationEvent event) {
			/* if any child turns valid, the whole form *might* be valid now */
			computeValidation();
			if (isValid()) {
				if (Html5FormPanel.this.submitButton != null) {
					Html5FormPanel.this.submitButton.setEnabled(true);
				}
			}
		}

	}

	private void fireJustBeforeFormSubmit() {
		for (final IFormSubmitEventHandler h : this.formSubmitEventHandlers) {
			h.onJustBeforeFormSubmit();
		}
	}

	private void fireOnFormSubmitSuccess() {
		unlock();
		for (final IFormSubmitEventHandler h : this.formSubmitEventHandlers) {
			h.onFormSubmitSuccess();
		}
	}

	private void fireOnFormSubmitFailed() {
		unlock();
		for (final IFormSubmitEventHandler h : this.formSubmitEventHandlers) {
			h.onFormSubmitFailed();
		}
	}

	public void init() {
	}

	/**
	 * @return true if the form can be submitted
	 */
	@Feature("validation")
	public boolean isValid() {
		return this.isValid.isTrue();
	}

	/**
	 * locking the form means locking all input elements and locking all buttons
	 */
	@Feature("locking")
	@Override
	public void lock() {
		for (final IHtml5EnabledInputWidget h5boc : this.textInputs) {
			h5boc.asHtml5TextInput().lock();
		}
		if (this.submitButton != null) {
			this.submitButton.lock();
		}
		LockUtils.lock(this);
	}

	/** AJAX: Form submit failed on server */
	@Override
	public void onFailure() {
		fireOnFormSubmitFailed();
	}

	@Override
	protected void onLoad() {
		super.onLoad();
		findAllChildrenAndAddThem(this);
		addStyleName("form");
		init();
	}

	/** AJAX: Form submit success on server */
	@Override
	public void onSuccess() {
		/* get ready for next submit */
		fireOnFormSubmitSuccess();
	}

	@Override
	protected void onUnload() {
		super.onUnload();
		for (final HandlerRegistration reg : this.validationHandlerRegistrations) {
			reg.removeHandler();
		}
		this.validationHandlerRegistrations.clear();
		this.textInputs.clear();
		this.submitButton = null;
	}

	/**
	 * Unlocks, sets fields to empty and resets validation state
	 */
	public void reset() {
		this.isValid = Is.Maybe;
		resetLockState();
		for (final IHtml5EnabledInputWidget h : this.textInputs) {
			h.asHtml5TextInput().reset();
		}
	}

	@Override
	@Feature("locking")
	public void resetLockState() {
		for (final IHtml5EnabledInputWidget h5boc : this.textInputs) {
			h5boc.asHtml5TextInput().resetLockState();
		}
		if (this.submitButton != null) {
			this.submitButton.resetLockState();
		}
		removeStyleName(CommonResourceBundle.INSTANCE.css().locked());
		// and:
		this.isSubmitting = false;
	}

	/**
	 * @param formSubmitter
	 *            responsible for executing the submit. The result must be send
	 *            to this form by calling {@link #onFailure()} or
	 *            {@link #onSuccess()}.
	 */
	public void setFormSubmitter(final IFormManagingWidget formSubmitter) {
		this.formSubmitter = formSubmitter;
	}

	/**
	 * For debug only
	 *
	 * @return
	 */
	private String stateAsString() {
		return "valid:" + this.isValid + " submitting:" + this.isSubmitting;
	}

	/**
	 * If form is valid and not already being submitted, it gets submitted.
	 *
	 * If an error occurs, the focus is moved to the first input that needs to
	 * be changed.
	 *
	 * @param source
	 *            who submits the form
	 */
	public void tryToSubmit(final IBelongsToHtml5Form source) {
		log.info("tryToSubmit state = " + stateAsString());

		final boolean incomplete = this.formSubmitEventHandlers.size() == 0;
		XyAssert.xyAssert(!incomplete, "No formEventHandler registered, cannot submit");

		if (!this.isValid.isTrue()) {
			computeValidation();
		}

		if (this.isValid.isTrue() && !this.isSubmitting) {
			this.isSubmitting = true;
			log.debug("Submitting form " + getElement().getId());
			fireJustBeforeFormSubmit();
			lock();
			this.formSubmitter.submitForm();
		} else {
			showCannotSubmitAndHelpToFix();
		}
	}

	private void showCannotSubmitAndHelpToFix() {
		/** Set focus on first field with error */
		boolean haveSetFocus = false;
		for (final IHtml5EnabledInputWidget h5box : this.textInputs) {
			assert h5box.asHtml5TextInput().computeValidation() != null : "should have been computed before";
			if (!h5box.asHtml5TextInput().computeValidation().level.isValid()) {
				// show validation errors
				h5box.asHtml5TextInput().activateValidationWarnings();

				log.info("Focussing input element with validation error");
				if (!haveSetFocus) {
					FocusUtils.setFocus(h5box.asHtml5TextInput().getTextBoxBase(), true);
					haveSetFocus = true;
				}
			}
		}
	}

	/**
	 * unlocking the form means unlocking all input elements and unlocking all
	 * buttons.
	 */
	@Feature("locking")
	@Override
	public void unlock() {
		for (final IHtml5EnabledInputWidget h5boc : this.textInputs) {
			h5boc.asHtml5TextInput().unlock();
		}
		if (this.submitButton != null) {
			this.submitButton.unlock();
		}
		LockUtils.unlock(this);
		// and:
		this.isSubmitting = false;
	}

	@Override
	public HandlerRegistration addInvalidationHandler(final InvalidationHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(InvalidationEvent.TYPE, this, handler);
	}

	@Override
	public HandlerRegistration addValidationHandler(final ValidationHandler handler) {
		return ClientApp.getEventBus().addHandlerToSource(ValidationEvent.TYPE, this, handler);
	}

}
