package com.calpano.common.client.view.forms.placeholder.impl;

import org.xydra.annotations.Feature;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.client.view.forms.Html5DomUtil;
import com.calpano.common.client.view.forms.locking.impl.H5Lockable;
import com.calpano.common.client.view.forms.placeholder.IPlaceholderSupport;
import com.calpano.common.client.view.forms.utils.KeyUtils;
import com.calpano.common.client.view.resources.CommonResourceBundle;
import com.calpano.common.client.view.resources.Css;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.TextBoxBase;

/**
 * Manages between state 'Placeholder' and 'Usertyped'.
 *
 * Fires a {@link ValueChangeEvent} when the user-typed text changed
 *
 * <h3>Listens to Events</h3> Listens to all {@link #PLACEHOLDER_EVENTS}, e.g.
 * Event.ONKEYDOWN, Event.ONKEYUP, Event.ONPASTE, Event.ONCHANGE, Event.ONCLICK,
 * Event.ONDBLCLICK, Event.ONFOCUS
 *
 * <h3>Fires events</h3> {@link ValueChangeEvent}
 *
 * <h3>On reset</h3> fire {@link ValueChangeEvent} if current text was not "";
 * mode=Placeholder;
 *
 * <h3>CSS attributes or JavaScript flags changed</h3> spellcheck,
 * {@link Css#placeholder()}
 *
 * @author xamde
 * @param <T>
 */
@Feature("placeholder")
public class H5Placeholder<T extends TextBoxBase & IPlaceholderSupport> extends H5Lockable<T>
		implements IPlaceholderSupport {

	public static enum State {
		/** showing a pre-defined placeholder text, e.g. in light gray */
		Placeholder,
		/** showing text typed by a user, usually in normal, e.g. black colour */
		Usertyped
	}

	private static final Logger log = LoggerFactory.getLogger(H5Placeholder.class);

	/** For which to listen to realize the placeholder feature */
	public static final int[] PLACEHOLDER_EVENTS = new int[] { Event.ONKEYDOWN, Event.ONKEYUP,
			Event.ONPASTE, Event.ONCHANGE, Event.ONCLICK, Event.ONDBLCLICK, Event.ONFOCUS };

	/** to be able to compute what happened */
	private String lastFiredValue = null;

	/** To be shown when field is empty and has no focus */
	private String placeholderText = "";

	/** enable or disable this feature via UiBinder */
	private final boolean simulatePlaceholder;

	/** Current state, initially showing a placeholder */
	private State state;

	/**
	 * @param base
	 * @param simulatePlaceholder
	 */
	public H5Placeholder(final T base, final boolean simulatePlaceholder) {
		super(base);
		CommonResourceBundle.INSTANCE.css().ensureInjected();
		this.simulatePlaceholder = simulatePlaceholder;
		setToInitialPlaceholderState();
	}

	/**
	 * @param userTypedText
	 *            changed value
	 */
	private void fireValueChangeEventIfValueChangeFromLastTime(final String userTypedText) {
		// don't fire if there is no change
		if (this.lastFiredValue != null && this.lastFiredValue.equals(userTypedText)) {
			return;
		}

		this.lastFiredValue = userTypedText;
		ValueChangeEvent.fire(getTextBoxBase(), userTypedText);
	}

	@Override
	protected int getEventsToSink() {
		int eventBits = 0;
		eventBits |= super.getEventsToSink();

		if (this.simulatePlaceholder) {
			for (int i = 0; i < PLACEHOLDER_EVENTS.length; i++) {
				eventBits |= PLACEHOLDER_EVENTS[i];
			}
		}
		return eventBits;
	}

	/**
	 * Gets the current placeholder text for the text box.
	 *
	 * @return the current placeholder text
	 */
	@Feature("placeholder")
	public String getPlaceholder() {
		return this.placeholderText;
	}

	@Override
	public String getRawText() {
		return this.base.getRawText();
	}

	@Feature("placeholder")
	public String getText() {
		return this.getUsertypedText();
	}

	public String getUsertypedText() {
		switch (this.state) {
		case Placeholder:
			return "";
		case Usertyped:
			return this.getRawText();
		}
		throw new AssertionError();
		//
		// String text = this.getRawText();
		// if(text.equals(getPlaceholder())) {
		// return "";
		// } else {
		// return text;
		// }
	}

	@SuppressWarnings("unused")
	private boolean hasNonEmptyPlaceholderText() {
		assert this.placeholderText != null;
		return this.placeholderText.length() > 0;
	}

	private void setToInitialPlaceholderState() {
		this.lastFiredValue = null;
		/* initially showing a placeholder */
		this.state = State.Placeholder;
	}

	private void moveCursorToFront() {
		/*
		 * there are some focus-quirks with setCursorPos: some browsers
		 * (FF&Chrome) re-focused the input automatically after this. Two ideas
		 * for fixing it: manually set focus (for the other browsers), or use
		 * some kind of deferred command to avoid loosing the focus in the first
		 * place.
		 */
		getTextBoxBase().setCursorPos(0);
		getTextBoxBase().setFocus(true);
	}

	/**
	 * If we listen to keydown events the browser does not give us the current
	 * text of the input: we get '' instead of 'placeholder' although we see the
	 * placeholder in the browser (visually). Therefore we use key-up.
	 *
	 * @param event
	 */
	protected void onBrowserEvent(final Event event) {
		if (!this.simulatePlaceholder) {
			return;
		}

		if (log.isTraceEnabled()) {
			log.trace("H5PlaceH event " + event.getType() + " in state = " + this.state);
		}

		final int eventType = DOM.eventGetType(event);

		if (this.state == State.Placeholder) {

			switch (eventType) {
			case Event.ONCLICK:
			case Event.ONDBLCLICK:
			case Event.ONFOCUS: {
				/* the only event for a drag-drop */

				/* don't let placeholder be selected */
				selectNothing();
				moveCursorToFront();

				/*
				 * drag & drop can put content in here. we ignore it and just
				 * auto-repair
				 */
				Scheduler.get().scheduleDeferred(new ScheduledCommand() {

					@Override
					public void execute() {
						placeholderTextRepairIfAltered();
					}
				});
			}
				break;
			case Event.ONKEYDOWN: {
				/**
				 * if event is fired on key-down, the text box does not yet
				 * contain the text after the user has typed. Therefore no event
				 * is fired if onKeyDown is true.
				 */

				if (KeyUtils.producesText(event.getKeyCode())) {
					switchToUserInputAndRemovePlaceholderText();
				} else {
					/* stop all events besides TAB (e.g. RETURN, Paste, ...) */
					if (event.getKeyCode() != KeyCodes.KEY_TAB) {
						event.stopPropagation();
						event.preventDefault();
						return;
					}
				}
				break;
			}
			case Event.ONPASTE: {
				/* only called for mouse-menu 'paste' */
				// IMPROVE keep pasted content, just trim it (and remove
				// placeholde text around it)
				switchToUserInputAndRemovePlaceholderText();
			}
				break;
			case Event.ONKEYUP: {
				/*
				 * 'tab'.key-up is fired in the field <em>after</em> focus
				 * switched
				 */
				if (event.getKeyCode() == KeyCodes.KEY_TAB) {
					onGetFocus();
				} else {
					if (KeyUtils.producesText(event.getKeyCode())) {
						fireValueChangeEventIfValueChangeFromLastTime(getRawText());
					}
				}
				return;
			}
			}
		} else {
			/* ================ UserTyped ============ */
			assert this.state == State.Usertyped;

			switch (eventType) {
			case Event.ONKEYUP:
				/* drag & drop sends ONLY an onFocus event */
			case Event.ONFOCUS:
			case Event.ONCHANGE: {
				final String value = this.getText();
				if (value.isEmpty()) {
					this.state = State.Placeholder;
					renderState();
					fireValueChangeEventIfValueChangeFromLastTime("");
				} else {
					fireValueChangeEventIfValueChangeFromLastTime(value);
				}
			}
				break;
			}
		}
	}

	protected void onGetFocus() {
		log.warn("getting focus");

		if (this.state == State.Usertyped) {
			selectAll();
		}
		moveCursorToFront();
	}

	@Override
	public void onLoad() {
		super.onLoad();
		if (this.simulatePlaceholder) {
			renderState();
		}
	}

	private void placeholderTextRepairIfAltered() {
		final String foundText = this.getRawText();
		if (!foundText.equals(getPlaceholder())) {
			log.info("autorepair placeholder text");
			placeholderTextShow();
		}
	}

	private void placeholderTextShow() {
		getTextBoxBase().setRawText(getPlaceholder());
		moveCursorToFront();
	}

	private void renderState() {
		switch (this.state) {
		case Placeholder: {
			getTextBoxBase().addStyleName(CommonResourceBundle.INSTANCE.css().placeholder());
			/* avoid distracting red underlines for mis-spelled placeholders */
			getTextBoxBase().getElement().setAttribute("spellcheck", "false");
			Scheduler.get().scheduleDeferred(new ScheduledCommand() {

				@Override
				public void execute() {
					placeholderTextShow();
				}
			});
		}
			break;
		case Usertyped: {
			getTextBoxBase().removeStyleName(CommonResourceBundle.INSTANCE.css().placeholder());
			/* spell-check user input */
			getTextBoxBase().getElement().removeAttribute("spellcheck");
			getTextBoxBase().setRawText("");
		}
		}
	}

	@Override
	protected void reset() {
		super.reset();
		if (!this.simulatePlaceholder) {
			return;
		}
		fireValueChangeEventIfValueChangeFromLastTime("");
		setToInitialPlaceholderState();
		renderState();
	}

	private void selectAll() {
		// works internally on getText() which can be empty
		getTextBoxBase().setSelectionRange(0, getTextBoxBase().getRawText().length());
	}

	private void selectNothing() {
		if (getTextBoxBase().getSelectionLength() > 0) {
			getTextBoxBase().setSelectionRange(0, 0);
		}
	}

	/**
	 * Called (indirectly) from UiBinder. HTML5 feature. Sets the placeholder
	 * text displayed in the text box.
	 *
	 * @param placeholderText
	 *            the placeholder text
	 */
	public void setPlaceholder(final String placeholderText) {
		this.placeholderText = placeholderText != null ? placeholderText : "";
		if (!this.simulatePlaceholder) {
			Html5DomUtil.setPropertyString(getTextBoxBase(), "placeholder", this.placeholderText);
		}
	}

	@Override
	public void setRawText(final String s) {
		getTextBoxBase().setRawText(s);
	}

	/**
	 * Sets text as if the user had typed it
	 *
	 * @param s
	 *            to be set
	 */
	public void setText(final String s) {
		if (log.isTraceEnabled()) {
			log.trace("setText " + s);
		}
		final String text = s != null ? s : "";

		switch (this.state) {
		case Placeholder: {
			if (text.length() > 0) {
				this.state = State.Usertyped;
				this.setRawText(text);
				renderState();
			}
		}
			break;
		case Usertyped: {
			if (text.length() == 0) {
				this.state = State.Placeholder;
				this.setRawText("");
				renderState();
			} else {
				this.setRawText(text);
				fireValueChangeEventIfValueChangeFromLastTime(text);
			}
		}
			break;
		default:
			throw new AssertionError();
		}
	}

	/**
	 * Sets text as if the user had typed it
	 *
	 * @param s
	 *            to be set
	 */
	public void setValue(final String s) {
		this.setText(s);
	}

	private void switchToUserInputAndRemovePlaceholderText() {
		if (this.state == State.Placeholder) {
			this.state = State.Usertyped;
			setRawText("");
			renderState();
		}
	}

	@SuppressWarnings("unused")
	private void trimContent() {
		getTextBoxBase().setRawText(getTextBoxBase().getRawText().trim());
	}

}
