package com.calpano.common.shared.data;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import com.calpano.common.shared.data.DataEvent.ActionKind;
import com.calpano.common.shared.data.DataEvent.DataEventHandler;
import com.google.web.bindery.event.shared.Event.Type;

/**
 * Used in DataEventWatcher.
 *
 * Defines a {@link DataEventCondition} consisting of a {@link Type} of
 * {@link DataEvent} and allowed {@link ActionKind}s.
 *
 * @author alpha
 */
public class DataEventCondition {

	/** any of these ActionKinds is ok, not all of them are required */
	private final Set<ActionKind> awaitedtActionKinds;

	/** expected event type */
	private final Type<DataEventHandler<?>> type;

	/**
	 * Created a new {@link DataEventCondition}
	 *
	 * @param type
	 *            {@link Type} of {@link DataEvent} to match against
	 * @param allowedKinds
	 *            set of allowed {@link ActionKind}s to match against
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	public <T> DataEventCondition(final Type<DataEventHandler<T>> type, final ActionKind... allowedKinds) {
		this.type = (Type) type;
		this.awaitedtActionKinds = new HashSet<ActionKind>(Arrays.asList(allowedKinds));
	}

	/**
	 * Returns the allowed {@link ActionKind}s
	 *
	 * @return type of {@link DataEvent}
	 */
	public Set<ActionKind> getAllowedKinds() {
		return this.awaitedtActionKinds;
	}

	/**
	 * Returns the type of {@link DataEvent}
	 *
	 * @return type of {@link DataEvent}
	 */
	public Type<DataEventHandler<?>> getAssociatedType() {
		return this.type;
	}

	/**
	 * Checks whether a given {@link DataEvent} matches this
	 * {@link DataEventCondition}, i.e., whether the {@link Type} of the event
	 * equals the specified {@link Type} in this condition and the event
	 * {@link ActionKind} is contained in the allowed kinds of this condition}.
	 *
	 * @param event
	 *            the {@link DataEvent} to check for a match
	 * @return true if the event matches this condition, else false
	 */
	public boolean matches(final DataEvent<?> event) {
		return getAssociatedType().equals(event.getAssociatedType())
				&& getAllowedKinds().contains(event.getKind());

	}

	@Override
	public String toString() {
		String out = "DataEventCondition: EventType=" + this.type.hashCode() + ",  awaited Kinds: ";
		for (final ActionKind kind : this.awaitedtActionKinds) {
			out += kind.name() + " ";
		}
		return out;
	}
}
