package com.calpano.common.shared.data;

/**
 * Basic states when booting and running an app: Booting, loading, syncing,
 * inSync and various error states.
 *
 * @author xamde
 */
public enum GlobalDataState {
	/**
	 * Initial state, always trying to load local data or init with built-ins.
	 * App might show a splash screen or some initial rendering without any
	 * data.
	 */
	Booting,
	/**
	 * All local data has been inspected and a stable state has been reached.
	 * The app should show some initial data, if appropriate. App should also
	 * show a spinning cursor to indicate further loading is in progress.
	 */
	LocalStable,
	/** Trying to reach server to get a partial or complete update */
	TryingToSync,
	/**
	 * Server did not answer in reasonable time
	 */
	Offline,
	/** Got some data from server, but we know more can be waiting */
	PartialUpdate,
	/**
	 * Got response from the server. Currently now more recent data is
	 * available.
	 */
	InSync,
	/**
	 * Some un-repair-able error occurred. The app should tell the user now what
	 * to do.
	 */
	SyncError;

	private transient GlobalDataState[] next;

	public boolean canProgressInto(final GlobalDataState nextState) {
		for (final GlobalDataState allowed : this.next) {
			if (allowed == nextState) {
				return true;
			}
		}
		return false;
	}

	static {
		/* define process model */
		GlobalDataState.Booting.next = new GlobalDataState[] { GlobalDataState.LocalStable };
		GlobalDataState.LocalStable.next = new GlobalDataState[] { GlobalDataState.TryingToSync };
		GlobalDataState.TryingToSync.next = new GlobalDataState[] { GlobalDataState.Offline,
				GlobalDataState.PartialUpdate, GlobalDataState.InSync, GlobalDataState.SyncError };
		GlobalDataState.Offline.next = new GlobalDataState[] { GlobalDataState.TryingToSync };
		GlobalDataState.PartialUpdate.next = new GlobalDataState[] { GlobalDataState.TryingToSync };
		GlobalDataState.InSync.next = new GlobalDataState[] { GlobalDataState.TryingToSync };
		// if there is some automatic retry going on
		GlobalDataState.SyncError.next = new GlobalDataState[] { GlobalDataState.TryingToSync };
	}

	public static GlobalDataState getInitial() {
		return GlobalDataState.Booting;
	}

}
