package com.calpano.common.shared.user;

import org.xydra.base.XId;
import org.xydra.base.rmof.XReadableObject;
import org.xydra.base.rmof.XWritableModel;

/**
 * A user that has been registered or only invited. Whenever a user A invites a
 * user B, there should be hops for both A and B.
 *
 * State should only be managed by
 * com.calpano.common.server.user.wrapper.ViralManager.
 *
 * Viral chain properties:
 * <ul>
 * <li>previousHopId</li>
 * <li>Set<XID> of successorHopsIds</li>
 * <li>directSuccesorCount</li>
 * <li>totelSuccessorCount</li>
 * <li>numberOfHopsFromRoot</li>
 * <li>offerId</li>
 * <li>secretKey</li>
 * </ul>
 *
 * @author xamde
 */
public interface IViralUser {

	/**
	 * @return the number of direct successors
	 */
	long getDirectSuccessorCount();

	/**
	 * @param successorHopSetObject
	 *            may be null
	 * @return Iterable<XID> for all successors IDs, never null
	 */
	Iterable<XId> getDirectSuccessorIds(XReadableObject successorHopSetObject);

	/**
	 * @return the distance from root. The root hop itself has a distance of 0.
	 *         The next hop has 1, and so on.
	 */
	long getNumberOfHopsFromRoot();

	/**
	 * @return the predecessor. For the root hop this is null.
	 */
	XId getPreviousHopId();

	/**
	 * @return the total successor count, i.e. all direct successors plus all
	 *         transitive successors of their successors.
	 */
	long getTotalSuccessorCount();

	boolean isRootUser();

	XId getId();

	/**
	 * Register an XID as a direct successor of this viral user.
	 *
	 * Note: Should only be used with txn semantics around it.
	 *
	 * @param model
	 *            must contain this object; in this model, a new proxy-object is
	 *            created; Field IDs in the proxy object are used to represent
	 *            successors of this ViralUser
	 * @param directSuccessor
	 */
	void addDirectSuccessorId(XWritableModel model, XId directSuccessor);

	/**
	 * Note: Should only be used with txn semantics around it.
	 *
	 * @param model
	 *            must contain this object; in this model, a new proxy-object is
	 *            created; Field IDs in the proxy object are used to represent
	 *            successors of this ViralUser
	 * @param directSuccessor
	 */
	void removeDirectSuccessorId(XWritableModel model, XId directSuccessor);

	/**
	 * Internal.
	 *
	 * @param directSuccesorCount
	 */
	void setDirectSuccesorCount(long directSuccesorCount);

	/**
	 * Internal.
	 *
	 * @param numberOfHopsFromRoot
	 */
	void setNumberOfHopsFromRoot(long numberOfHopsFromRoot);

	/**
	 * Internal. Who was the origin of this hop?
	 *
	 * @param previousHopId
	 */
	void setPreviousHopId(XId previousHopId);

	/**
	 * Internal.
	 *
	 * @param totalSuccessorCount
	 */
	void setTotalSuccessorCount(long totalSuccessorCount);

	XId getObjectIdToLoad();

}
