package com.calpano.common.shared.user.base;

import org.xydra.annotations.RunsInGWT;
import org.xydra.annotations.Template;
import org.xydra.base.XId;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.core.model.XObject;

import com.calpano.common.shared.mail.IMailUser;
import com.calpano.common.shared.user.INameEmailUser;
import com.calpano.common.shared.user.wrapper.AuthenticationUser;
import com.calpano.common.shared.user.wrapper.CoreUser;
import com.calpano.common.shared.user.wrapper.IdentificationUser;
import com.calpano.common.shared.user.wrapper.ManagedUser;
import com.calpano.common.shared.user.wrapper.MetricUser;
import com.calpano.common.shared.user.wrapper.ViralUser;
import com.calpano.common.shared.xydrautils.IBasedOnXWritableObject;
import com.calpano.common.shared.xydrautils.TaggedObject;
import com.calpano.common.shared.xydrautils.field.FieldProperty;
import com.calpano.common.shared.xydrautils.impl.BasedOnXWritableObject;

import de.xam.p13n.shared.Personalisation;

/**
 * Contains all data required for identification.
 *
 * A basic shared user that is used in a xydra context.
 *
 * Identification properties:
 * <ul>
 * <li>email address</li>
 * <li>alternate email addresses</li>
 * <li>username</li>
 * </ul>
 *
 * All data is stored in an {@link XObject}.
 *
 * @author xamde
 */
@RunsInGWT(true)
public class SharedBaseUser extends BasedOnXWritableObject implements IBasedOnXWritableObject,
		INameEmailUser, IMailUser {

	public static FieldProperty<String> _fullname = new FieldProperty<String>("fullname",
			String.class);

	private static final long serialVersionUID = 1L;

	private transient IdentificationUser identificationUser;

	private transient CoreUser coreUser;

	private transient MetricUser metricUser;

	private transient AuthenticationUser authUser;

	private transient ViralUser viraluser;

	private transient ManagedUser managedUser;

	/** GWT only */
	protected SharedBaseUser() {
		super();
	}

	public SharedBaseUser(final XWritableObject writableObject, final XId actorId) {
		super(writableObject, actorId);
	}

	public SharedBaseUser(final IBasedOnXWritableObject proto) {
		super(proto);
	}

	public SharedBaseUser(final XWritableObject writableObject, final XId actorId, final boolean isJustCreated) {
		super(writableObject, actorId);
		this.isJustCreated = isJustCreated;
	}

	public SharedBaseUser(final SharedBaseUser user, final boolean justCreated) {
		super(user);
		this.isJustCreated = justCreated;
	}

	private transient boolean isJustCreated = false;

	/**
	 * @return true if the object has just been created. Default and most common
	 *         case is false = object has been loaded. The exact semantics of
	 *         'created' make only sense in a session context. This property is
	 *         not persisted or serialised.
	 */
	public boolean isJustCreated() {
		return this.isJustCreated;
	}

	public IdentificationUser asIdentificationUser() {
		if (this.identificationUser == null) {
			this.identificationUser = new IdentificationUser(this);
		}
		return this.identificationUser;
	}

	@Override
	@Template("Real world user name")
	public String getName() {
		return _fullname.getValue(getXObject());
	}

	@Override
	public String getNameOrEmail() {
		return NameEmailUtils.getNameOrEmail(getName(), getEmail());
	}

	@Override
	public String getNameAndEmail() {
		return NameEmailUtils.getNameAndEmail(getName(), getEmail());
	}

	/**
	 * @param name
	 *            the real world name of the user, e.g. 'John Doe'. @CanBeNull
	 *            to remove any existing value
	 */
	public void setName(final String name) {
		_fullname.setValue(getActorId(), getXObject(), name);
	}

	@Override
	public String getEmail() {
		return asIdentificationUser().getEmail();
	}

	public CoreUser asCoreUser() {
		if (this.coreUser == null) {
			this.coreUser = new CoreUser(this);
		}
		return this.coreUser;
	}

	public MetricUser asMetricUser() {
		if (this.metricUser == null) {
			this.metricUser = new MetricUser(this);
		}
		return this.metricUser;
	}

	public ViralUser asViralUser() {
		if (this.viraluser == null) {
			this.viraluser = new ViralUser(this);
		}
		return this.viraluser;
	}

	public AuthenticationUser asAuthenticationUser() {
		if (this.authUser == null) {
			this.authUser = new AuthenticationUser(this);
		}
		return this.authUser;
	}

	public ManagedUser asManagedUser() {
		if (this.managedUser == null) {
			this.managedUser = new ManagedUser(this);
		}
		return this.managedUser;
	}

	/**
	 * @return this instance as a {@link TaggedObject}. Never null.
	 */
	@Template("tags")
	public synchronized TaggedObject asTaggedObject() {
		return new TaggedObject(this);
	}

	@Override
	public void setLastMailSent(final long lastMailSent) {
		asMetricUser().setLastMailSent(lastMailSent);

	}

	@Override
	public long getFirstMailSent() {
		return asMetricUser().getFirstMailSent();
	}

	@Override
	public void setFirstMailSent(final long firstMailSent) {
		asMetricUser().setFirstMailSent(firstMailSent);
	}

	@Override
	public Personalisation getPersonalisation() {
		return asCoreUser().getPersonalisation();
	}

}
