package com.calpano.common.shared.validation;

import org.xydra.annotations.RunsInGWT;

/**
 * A single validation message.
 *
 * @author xamde
 */
@RunsInGWT(true)
public class ValidationMessage {

	/**
	 * Most widgets wont contain text at Info level.
	 */
	public enum Level {

		/** content is valid */
		OK(true, false),
		/**
		 * content is valid, but not a good idea, i.e. a bad password, an
		 * unusual name
		 */
		Warn(true, false),
		/**
		 * content is invalid, but really it is just missing where some content
		 * is required.
		 */
		ErrorMissing(false, false),
		/**
		 * content is invalid. There is non-empty content, and it is the wrong
		 * one
		 */
		ErrorWrong(false, true);

		private boolean valid;
		private boolean wrongContent;

		Level(final boolean valid, final boolean wrongContent) {
			this.valid = valid;
			this.wrongContent = wrongContent;
		}

		public boolean isValid() {
			return this.valid;
		}

		/**
		 * @return true if the element truly contains wrong content. False if it
		 *         contains just content although a content is required.
		 */
		public boolean hasWrongContent() {
			return this.wrongContent;
		}

		public boolean moreCriticalThan(final Level other) {
			return other == null || compareTo(other) > 0;
		}

		public boolean atLeastAsCriticalAs(final Level other) {
			return other == null || compareTo(other) >= 0;
		}

		public boolean lessCriticalThan(final Level other) {
			return other != null && compareTo(other) < 0;
		}

	}

	public static void main(final String[] args) {
		assert Level.ErrorMissing.moreCriticalThan(Level.OK);
		assert Level.OK.lessCriticalThan(Level.ErrorMissing);
		assert Level.ErrorMissing.atLeastAsCriticalAs(Level.OK);
		assert Level.ErrorMissing.atLeastAsCriticalAs(Level.ErrorMissing);

		assert Level.ErrorMissing.moreCriticalThan(null);
		assert !Level.OK.lessCriticalThan(null);
		assert Level.ErrorMissing.atLeastAsCriticalAs(null);
	}

	@Override
	public String toString() {
		return "[" + this.level.name() + "]='" + this.message + "'";
	}

	/** Built-in message if there is no error */
	public static final ValidationMessage ALL_OK = new ValidationMessage(Level.OK, "OK");

	public final Level level;

	/**
	 * Can be a text ready to be displayed or the key of an I18N property.
	 *
	 * If the widget is invalid, the text should explain the user how to make it
	 * valid.
	 */
	public final String message;

	public ValidationMessage(final Level level, final String message) {
		this.level = level;
		this.message = message;
	}

	public static ValidationMessage error(final String msg, final boolean wrongContent) {
		return new ValidationMessage(wrongContent ? Level.ErrorWrong : Level.ErrorMissing, msg);
	}

}
