package com.calpano.common.shared.xydrautils;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.xydra.base.XId;
import org.xydra.base.rmof.XWritableObject;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;

import com.calpano.common.shared.xydrautils.field.StringSetFieldProperty;
import com.calpano.common.shared.xydrautils.impl.BasedOnXWritableObject;

/**
 * Wraps any {@link XWritableObject} to support adding, removing to a string set
 *
 * @author xamde
 */
public class StringSetWrapper extends BasedOnXWritableObject implements Set<String> {

	private static final long serialVersionUID = 1L;

	@SuppressWarnings("unused")
	private static final Logger log = LoggerFactory.getLogger(StringSetWrapper.class);

	public StringSetWrapper(final XWritableObject xo, final XId actorId, final String fieldId) {
		super(xo, actorId);
		this.stringSetProperty = new StringSetFieldProperty(fieldId);
	}

	public StringSetWrapper(final IBasedOnXWritableObject proto, final String fieldId) {
		super(proto);
		this.stringSetProperty = new StringSetFieldProperty(fieldId);
	}

	private final StringSetFieldProperty stringSetProperty;

	/**
	 * @param value
	 */
	public void addValue(final String value) {
		this.stringSetProperty.addToCollection(getActorId(), getXObject(), value);
	}

	/**
	 * @return never null
	 */
	public Set<String> getValues() {
		return this.stringSetProperty.getValue(getXObject());
	}

	public boolean hasValue(final String value) {
		return getValues().contains(value);
	}

	public void removeValue(final String value) {
		this.stringSetProperty.removeFromCollection(getActorId(), getXObject(), value);
	}

	@Override
	public int size() {
		return getValues().size();
	}

	@Override
	public boolean isEmpty() {
		return getValues().isEmpty();
	}

	@Override
	public boolean contains(final Object o) {
		return o instanceof String && hasValue((String) o);
	}

	@Override
	public Iterator<String> iterator() {
		return getValues().iterator();
	}

	@Override
	public Object[] toArray() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public <T> T[] toArray(final T[] a) {
		return getValues().toArray(a);
	}

	@Override
	public boolean add(final String e) {
		final boolean b = hasValue(e);
		addValue(e);
		return !b;
	}

	@Override
	public boolean remove(final Object o) {
		final boolean b = contains(o);

		if (o instanceof String) {
			removeValue((String) o);
		}
		return b;
	}

	@Override
	public boolean addAll(final Collection<? extends String> c) {
		boolean anyChange = false;
		for (final String s : c) {
			anyChange |= add(s);
		}
		return anyChange;
	}

	@Override
	public boolean containsAll(final Collection<?> c) {
		for (final Object o : c) {
			if (!(o instanceof String) || !hasValue((String) o)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public boolean retainAll(final Collection<?> c) {
		final List<String> toBeRemoved = new LinkedList<String>();
		for (final String s : this) {
			if (!c.contains(s)) {
				toBeRemoved.add(s);
			}
		}
		return removeAll(toBeRemoved);
	}

	@Override
	public boolean removeAll(final Collection<?> c) {
		boolean anyChange = false;
		for (final Object o : c) {
			anyChange |= remove(o);
		}
		return anyChange;
	}

	@Override
	public void clear() {
		this.stringSetProperty.clear(getXObject());
	}

}
