package de.xam.cds;

import org.xydra.index.ITripleSink;

/**
 * Type-safe version of CDS names.
 *
 * This class offers no access to pre-define nice human readable labels. Use {@link CdsId#getLabel()} for that.
 *
 * See {@link CdsIdTGenerator_NoGwt} for some code parts.
 *
 * See also {@link CdsId}
 *
 * @author xamde
 *
 * @param <T>
 */
public class CdsIdT<T> {

	/**
	 * Defines the mapping from a CdsId to a type-safe T
	 *
	 * @author xamde
	 *
	 * @param <T>
	 */
	public static interface IIdMapper<T> {

		/**
		 * @param cdsId
		 * @return a type-safe T representing the same id
		 */
		T toId(CdsId cdsId);
	}

	private final IIdMapper<T> mapper;

	public IIdMapper<T> getMapper() {
		return this.mapper;
	}

	public CdsIdT(final IIdMapper<T> mapper) {
		this.mapper = mapper;
		init();
	}

	// start of 'generated' code

	/** 'has inverse' */
	public T hasInverse;

	/** 'transitive relation' */
	public T type_transitiveRelation;

	/** 'symmetric relation' */
	public T type_symmetricRelation;

	/** 'is related to' */
	public T hasRelated;

	/** 'links to' */
	public T hasTarget;

	/** Added on 2014-07-13 */
	public T hasManualLinkTarget;

	/** Added on 2014-07-13 */
	public T hasAutoLinkTarget;

	/** 'is linked from' */
	public T hasSource;

	/** Added on 2014-07-13 */
	public T hasManualLinkSource;

	/** Added on 2014-07-13 */
	public T hasAutoLinkSource;

	/** 'is similar to' */
	public T hasSimilar;

	/** 'annotates' */
	public T hasAnnotationMember;

	/** ' has annotation' */
	public T hasAnnotation;

	/** 'comes before' */
	public T hasAfter;

	/** 'comes after' */
	public T hasBefore;

	/** 'has detail' */
	public T hasDetail;

	/** 'has context' */
	public T hasContext;

	/** 'is same as' */
	public T sameAs;

	/** 'replaces' */
	public T replaces;

	/** 'is replaced by' */
	public T replacedBy;

	/** 'is alias for' */
	public T isAliasOf;

	/** 'has alias' */
	public T hasAlias;

	/** 'is tag of' */
	public T hasTagMember;

	/** 'is tagged with' */
	public T hasTag;

	/** 'has subType' */
	public T hasSubType;

	/** 'is subtype of' */
	public T hasSuperType;

	/** 'has part' */
	public T hasPart;

	/** 'is part of' */
	public T isPartOf;

	/** 'has instance' */
	public T hasInstance;

	/** 'has type' */
	public T hasType;

	/** 'relation' */
	public T type_relation;

	@SuppressWarnings("unchecked")
	public T[] values() {
		return (T[]) new Object[] { this.hasAfter, this.hasAlias, this.hasAnnotation, this.hasAnnotationMember,
				this.hasBefore, this.hasContext, this.hasDetail, this.hasInstance, this.hasInverse, this.hasPart,
				this.hasRelated, this.hasSimilar, this.hasSource, this.hasSubType, this.hasSuperType, this.hasTag,
				this.hasTagMember, this.hasTarget, this.hasType, this.isAliasOf, this.isPartOf, this.replacedBy,
				this.replaces, this.sameAs, this.type_transitiveRelation, this.type_symmetricRelation,
				this.type_relation,

				this.hasAutoLinkSource, this.hasAutoLinkTarget, this.hasManualLinkTarget, this.hasManualLinkSource };
	}

	private boolean initDone;

	/**
	 * Defines the exact id strings. By convention, they should match the enum names from {@link CdsId} 1:1.
	 */
	private synchronized void init() {
		if (this.initDone) {
			return;
		}

		/* for GWT compatibility, no Java reflection is used here */

		this.hasInverse = this.mapper.toId(CdsId.hasInverse);
		this.type_transitiveRelation = this.mapper.toId(CdsId.type_transitiveRelation);
		this.type_symmetricRelation = this.mapper.toId(CdsId.type_symmetricRelation);
		this.hasRelated = this.mapper.toId(CdsId.hasRelated);
		this.hasTarget = this.mapper.toId(CdsId.hasTarget);
		this.hasSource = this.mapper.toId(CdsId.hasSource);
		this.hasSimilar = this.mapper.toId(CdsId.hasSimilar);
		this.hasAnnotationMember = this.mapper.toId(CdsId.hasAnnotationMember);
		this.hasAnnotation = this.mapper.toId(CdsId.hasAnnotation);
		this.hasAfter = this.mapper.toId(CdsId.hasAfter);
		this.hasBefore = this.mapper.toId(CdsId.hasBefore);
		this.hasDetail = this.mapper.toId(CdsId.hasDetail);
		this.hasContext = this.mapper.toId(CdsId.hasContext);
		this.hasAutoLinkTarget = this.mapper.toId(CdsId.hasAutoLinkTarget);
		this.hasAutoLinkSource = this.mapper.toId(CdsId.hasAutoLinkSource);
		this.hasManualLinkTarget = this.mapper.toId(CdsId.hasManualLinkTarget);
		this.hasManualLinkSource = this.mapper.toId(CdsId.hasManualLinkSource);
		this.sameAs = this.mapper.toId(CdsId.sameAs);
		this.replaces = this.mapper.toId(CdsId.replaces);
		this.replacedBy = this.mapper.toId(CdsId.replacedBy);
		this.isAliasOf = this.mapper.toId(CdsId.isAliasOf);
		this.hasAlias = this.mapper.toId(CdsId.hasAlias);
		this.hasTagMember = this.mapper.toId(CdsId.hasTagMember);
		this.hasTag = this.mapper.toId(CdsId.hasTag);
		this.hasSubType = this.mapper.toId(CdsId.hasSubType);
		this.hasSuperType = this.mapper.toId(CdsId.hasSuperType);
		this.hasPart = this.mapper.toId(CdsId.hasPart);
		this.isPartOf = this.mapper.toId(CdsId.isPartOf);
		this.hasInstance = this.mapper.toId(CdsId.hasInstance);
		this.hasType = this.mapper.toId(CdsId.hasType);
		this.type_relation = this.mapper.toId(CdsId.type_relation);
		assert this.hasRelated != null;

		this.initDone = true;
	}

	/**
	 * Adds all built-in CDS axioms into the given triple index.
	 *
	 * @param tripleSink
	 */
	public void addAxiomaticTriples(final ITripleSink<T, T, T> tripleSink) {
		assert tripleSink != null;
		assert this.hasRelated != null;

		/** inverse relations and relationship hierarchy */

		// root relation
		addSoloRelation(tripleSink, this.hasRelated, this.hasRelated);

		// relation hierarchy
		addRelation(this.hasRelated, tripleSink, this.hasSimilar, this.hasSimilar);
		addRelation(this.hasRelated, tripleSink, this.hasTarget, this.hasSource);
		addRelation(this.hasTarget, tripleSink, this.hasAfter, this.hasBefore);
		addRelation(this.hasTarget, tripleSink, this.hasAnnotationMember, this.hasAnnotation);
		addRelation(this.hasTarget, tripleSink, this.hasDetail, this.hasContext);
		addRelation(this.hasTagMember, tripleSink, this.hasInstance, this.hasType);
		addRelation(this.hasDetail, tripleSink, this.hasPart, this.isPartOf);
		addRelation(this.hasDetail, tripleSink, this.hasSubType, this.hasSuperType);
		addRelation(this.hasAnnotationMember, tripleSink, this.hasTagMember, this.hasTag);
		addRelation(this.hasSimilar, tripleSink, this.isAliasOf, this.hasAlias);
		addRelation(this.hasSimilar, tripleSink, this.replaces, this.replacedBy);
		addRelation(this.hasSimilar, tripleSink, this.sameAs, this.sameAs);
		/** Added on 2014-07-13 */
		addRelation(this.hasTarget, tripleSink, this.hasAutoLinkTarget, this.hasAutoLinkSource);
		/** Added on 2014-07-13 */
		addRelation(this.hasTarget, tripleSink, this.hasManualLinkTarget, this.hasManualLinkSource);

		// relation types -- see CdsId.isTransitive
		tripleSink.index(this.hasAfter, this.hasType, this.type_transitiveRelation);
		tripleSink.index(this.hasBefore, this.hasType, this.type_transitiveRelation);
		tripleSink.index(this.sameAs, this.hasType, this.type_transitiveRelation);
		tripleSink.index(this.hasSubType, this.hasType, this.type_transitiveRelation);
		tripleSink.index(this.hasSuperType, this.hasType, this.type_transitiveRelation);

		tripleSink.index(this.hasInverse, this.hasType, this.type_symmetricRelation);
		tripleSink.index(this.sameAs, this.hasType, this.type_symmetricRelation);
		tripleSink.index(this.hasRelated, this.hasType, this.type_symmetricRelation);

		// added 2015-04-07; removed again on 2014-09-13
		// tripleSink.index(this.hasRelated, this.hasSubType, this.type_symmetricRelation);
		// tripleSink.index(this.hasRelated, this.hasSubType, this.type_transitiveRelation);

		// added 2015-09-13
		tripleSink.index(this.type_relation, this.hasSubType, this.type_symmetricRelation);
		tripleSink.index(this.type_relation, this.hasSubType, this.type_transitiveRelation);
	}

	/**
	 * Internal access only.
	 *
	 * Adds a relation without a super-relation.
	 *
	 * @param tripleSink
	 * @NeverNull
	 * @param relation
	 * @NeverNull
	 * @param inverseRelation
	 * @NeverNull
	 */
	private void addSoloRelation(final ITripleSink<T, T, T> tripleSink, final T relation, final T inverseRelation) {
		addSoloRelation(tripleSink, relation, inverseRelation, this.hasInverse, this.hasType, this.type_relation);
	}

	/**
	 * Adds a relation without a super-relation (edge case).
	 *
	 * @param tripleSink
	 * @NeverNull
	 * @param relation
	 * @NeverNull
	 * @param inverseRelation
	 * @NeverNull
	 * @param hasInverse
	 * @param hasType
	 * @param type_relation
	 */
	private static <T> void addSoloRelation(final ITripleSink<T, T, T> tripleSink, final T relation,
			final T inverseRelation, final T hasInverse, final T hasType, final T type_relation) {
		tripleSink.index(relation, hasInverse, inverseRelation);
		tripleSink.index(relation, hasType, type_relation);
		tripleSink.index(inverseRelation, hasType, type_relation);
	}

	/**
	 * Internal access only.
	 *
	 * Adds a relation with a super-relation.
	 *
	 * @param superRelation
	 * @NeverNull
	 * @param tripleSink
	 * @NeverNull
	 * @param relation
	 * @NeverNull
	 * @param inverseRelation
	 * @NeverNull
	 */
	private void addRelation(final T superRelation, final ITripleSink<T, T, T> tripleSink, final T relation,
			final T inverseRelation) {
		addRelation(tripleSink, relation, inverseRelation, superRelation, this.hasInverse, this.hasType,
				this.type_relation, this.hasSubType);
	}

	/**
	 * Adds a relation with a super-relation (normal case).
	 *
	 * @param tripleSink
	 * @NeverNull
	 * @param relation
	 * @NeverNull
	 * @param inverseRelation
	 * @NeverNull
	 * @param superRelation
	 * @param hasInverse
	 * @param hasType
	 * @param type_relation
	 * @param hasSubType
	 */
	private static <T> void addRelation(final ITripleSink<T, T, T> tripleSink, final T relation,
			final T inverseRelation, final T superRelation,

	final T hasInverse, final T hasType, final T type_relation, final T hasSubType) {
		addSoloRelation(tripleSink, relation, inverseRelation, hasInverse, hasType, type_relation);
		tripleSink.index(superRelation, hasSubType, relation);
	}
}
