package de.xam.cds;

import de.xam.cds.CdsIdT.IIdMapper;
import de.xam.triplerules.IRuleEngine;
import de.xam.triplerules.ITriplePattern;
import de.xam.triplerules.IVariable;
import de.xam.triplerules.impl.ConditionBinding;
import de.xam.triplerules.impl.ICostEstimator;
import de.xam.triplerules.impl.RuleEngine;
import de.xam.triplerules.impl.RuleUtils;
import de.xam.triplerules.impl.TriplePattern;
import de.xam.triplerules.impl.TripleRule;

/**
 * The built-in CDS inference rules. Work together with matching axiomatic
 * triples in {@link CdsIdT}.
 *
 * Impl note: Things are not static because generics in CdsIdT and the IAdapter
 *
 * @author xamde
 *
 * @param <T> id type
 */
public class CdsRules<T> implements ICostEstimator<T, T, T> {

	/**
	 * <pre>
	 * ?x   ?p        ?y
	 * ?y   ?p        ?z
	 * ?p   'hasType' 'transitiveRelation'
	 *   ==>
	 * ?x   ?p        ?z
	 * </pre>
	 */
	private final TripleRule<T, T, T> r01_transitive = new TripleRule<T, T, T>("r01-transitive");

	/**
	 *
	 * <pre>
	 * ?x   ?p             ?y
	 * ?p   'hasInverse'   ?q
	 *   ==>
	 * ?y   ?q             ?x
	 * </pre>
	 */
	private final TripleRule<T, T, T> r02_inverseA = new TripleRule<T, T, T>("r02-inverse-A");

	/**
	 * This rule is not active. The axiomatic triple ('hasInverse' 'hasType'
	 * 'symmetricRelation') is used instead
	 *
	 * <pre>
	 * ?p   'hasInverse'   ?q
	 *   ==>
	 * ?q   'hasInverse'   ?p
	 * </pre>
	 */
	private final TripleRule<T, T, T> r03_inverseT = new TripleRule<T, T, T>("r03-inverse-T");

	/**
	 * <pre>
	 * ?x   'hasType'        ?t
	 * ?t   'hasSuperType'   ?u
	 *   ==>
	 * ?x   'hasType'        ?u
	 * </pre>
	 */
	private final TripleRule<T, T, T> r04_type_inheritance = new TripleRule<T, T, T>(
			"r04-type-inheritance");

	/**
	 * <pre>
	 * ?x   ?p               ?y
	 * ?p   'hasSuperType'   ?u
	 *   ==>
	 * ?x   ?u               ?y
	 * </pre>
	 */
	private final TripleRule<T, T, T> r05_rel_inheritance = new TripleRule<T, T, T>(
			"r05-relation-inheritance");

	/**
	 * <pre>
	 * ?alias    'isAliasOf' ?resolved
	 * ?alias    ?p          ?o
	 *   ==>
	 * ?resolved ?p          ?o
	 * </pre>
	 */
	private final TripleRule<T, T, T> r06_alias_s = new TripleRule<T, T, T>("r06-alias_s");

	/**
	 * <pre>
	 * ?alias    'isAliasOf' ?resolved
	 * ?s        ?alias      ?o
	 *   ==>
	 * ?s        ?resolved   ?o
	 * </pre>
	 */
	private final TripleRule<T, T, T> r07_alias_p = new TripleRule<T, T, T>("r07-alias_p");

	/**
	 * <pre>
	 * ?alias    'isAliasOf' ?resolved
	 * ?s        ?p          ?alias
	 *   ==>
	 * ?s        ?p          ?resolved
	 * </pre>
	 */
	private final TripleRule<T, T, T> r08_alias_o = new TripleRule<T, T, T>("r08-alias_o");

	/**
	 * <pre>
	 * ?x   ?p           ?y
	 * ?p   'hasType'    'symmetricRelation'
	 *   ==>
	 * ?y   ?p           ?x
	 * </pre>
	 */
	private final TripleRule<T, T, T> r09_symmetric_A = new TripleRule<T, T, T>("r09-symmetric-A");

	/**
	 * <pre>
	 * ?p   'hasInverse' ?p
	 *   ==>
	 * ?p   'hasType'    'symmetricRelation'
	 * </pre>
	 */
	private final TripleRule<T, T, T> r10_selfInverse_symmetric = new TripleRule<T, T, T>(
			"r10_selfInverse_symmetric");

	/**
	 * <pre>
	 * ?p   'hasType'    'symmetricRelation'
	 *   ==>
	 * ?p   'hasInverse' ?p
	 * </pre>
	 */
	private final TripleRule<T, T, T> r11_symmetric_selfInverse = new TripleRule<T, T, T>(
			"r11_symmetric_selfInverse");

	/**
	 * <pre>
	 * ?p   'hasInverse' ?q
	 * ?q   'hasInverse' ?r
	 *   ==>
	 * ?p   'sameAs'     ?r
	 * </pre>
	 */
	private final TripleRule<T, T, T> r12_inverseCausesSameAs = new TripleRule<T, T, T>(
			"r12_inverseCausesSameAs");

	/**
	 * <pre>
	 * ?p   'hasInverse'   ?q
	 * ?p   'hasType'      ?t
	 *   ==>
	 * ?q   'hasType'      ?t
	 * </pre>
	 */
	private final TripleRule<T, T, T> r13_inverseType = new TripleRule<T, T, T>("r13-inverseType");

	/**
	 * <pre>
	 * ?a       'sameAs'     ?b
	 * ?a        ?p          ?c
	 *   ==>
	 * ?b        ?p          ?c
	 * </pre>
	 */
	private final TripleRule<T, T, T> r14_sameAs_s = new TripleRule<T, T, T>("r14-sameAs-s");

	/**
	 * <pre>
	 * ?p       'sameAs'     ?r
	 * ?a        ?p          ?b
	 *   ==>
	 * ?a        ?r          ?b
	 * </pre>
	 */
	private final TripleRule<T, T, T> r15_sameAs_p = new TripleRule<T, T, T>("r15-sameAs-p");

	/**
	 * <pre>
	 * ?b       'sameAs'     ?c
	 * ?a        ?p          ?b
	 *   ==>
	 * ?a        ?p          ?c
	 * </pre>
	 */
	private final TripleRule<T, T, T> r16_sameAs_o = new TripleRule<T, T, T>("r16-sameAs-o");

	/**
	 * <pre>
	 * ?a		'hasInverse'	?b
	 * ?c		'hasInverse'	?d
	 * ?a		'isSubtypeOf'   ?c
	 *   ==>
	 * ?b       'isSubtypeOf'   ?d
	 * </pre>
	 */
	private final TripleRule<T, T, T> r17_inv_subType_inv = new TripleRule<T, T, T>(
			"r17-inv-subType-inv");

	private final CdsIdT<T> cds;

	public CdsRules(final IIdMapper<T> mapper) {
		this.cds = new CdsIdT<T>(mapper);
	}

	private void addAliasRules(final IRuleEngine<T, T, T> engine) {
		this.r06_alias_s
				.condition()

				.addCondition(
						new TriplePattern<T, T, T>("alias", null, "fix1", this.cds.isAliasOf,
								"resolved", null))
				.addCondition(new TriplePattern<T, T, T>("alias", null, "p", null, "o", null));
		this.r06_alias_s.action().addAction(
				new TriplePattern<T, T, T>("resolved", null, "p", null, "o", null));
		engine.addRule(this.r06_alias_s);

		this.r07_alias_p
				.condition()

				.addCondition(
						new TriplePattern<T, T, T>("alias", null, "fix1", this.cds.isAliasOf,
								"resolved", null))
				.addCondition(new TriplePattern<T, T, T>("s", null, "alias", null, "o", null));
		this.r07_alias_p.action().addAction(
				new TriplePattern<T, T, T>("s", null, "resolved", null, "o", null));
		engine.addRule(this.r07_alias_p);

		this.r08_alias_o
				.condition()

				.addCondition(
						new TriplePattern<T, T, T>("alias", null, "fix1", this.cds.isAliasOf,
								"resolved", null))
				.addCondition(new TriplePattern<T, T, T>("s", null, "p", null, "alias", null));
		this.r08_alias_o.action().addAction(
				new TriplePattern<T, T, T>("s", null, "p", null, "resolved", null));
		engine.addRule(this.r08_alias_o);
	}

	private void addSameAsRules(final IRuleEngine<T, T, T> engine) {
		this.r14_sameAs_s.condition()

		.addCondition(new TriplePattern<T, T, T>("a", null, "fix1", this.cds.sameAs, "b", null))
				.addCondition(new TriplePattern<T, T, T>("a", null, "p", null, "c", null));
		this.r14_sameAs_s.action().addAction(
				new TriplePattern<T, T, T>("b", null, "p", null, "c", null));
		engine.addRule(this.r14_sameAs_s);

		this.r15_sameAs_p.condition()

		.addCondition(new TriplePattern<T, T, T>("p", null, "fix1", this.cds.sameAs, "r", null))
				.addCondition(new TriplePattern<T, T, T>("a", null, "p", null, "b", null));
		this.r15_sameAs_p.action().addAction(
				new TriplePattern<T, T, T>("a", null, "r", null, "b", null));
		engine.addRule(this.r15_sameAs_p);

		this.r16_sameAs_o.condition()

		.addCondition(new TriplePattern<T, T, T>("b", null, "fix1", this.cds.sameAs, "c", null))
				.addCondition(new TriplePattern<T, T, T>("a", null, "p", null, "b", null));
		this.r16_sameAs_o.action().addAction(
				new TriplePattern<T, T, T>("a", null, "p", null, "c", null));
		engine.addRule(this.r16_sameAs_o);
	}

	public void addAxiomaticRulesForCDS(final IRuleEngine<T, T, T> engine) {
		addTransitiveRules(engine);
		addAxiomaticRulesForCDS_exceptTransitives(engine);

		assert RuleUtils.couldTrigger(this.r05_rel_inheritance, this.r01_transitive);

	}

	public void addAxiomaticRulesForCDS_exceptTransitives(final IRuleEngine<T, T, T> engine) {
		addInverseRules(engine);
		addAliasRules(engine);
		addSameAsRules(engine);
		addTypeInheritanceRules(engine);
		addSymmetricRules(engine);

		assert !RuleUtils.couldTrigger(this.r03_inverseT, this.r04_type_inheritance);
	}

	private void addSymmetricRules(final IRuleEngine<T, T, T> engine) {
		this.r09_symmetric_A
				.condition()

				.addCondition(new TriplePattern<T, T, T>("x", null, "p", null, "y", null))
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasType, "fix2",
								this.cds.type_symmetricRelation));
		this.r09_symmetric_A.action().addAction(
				new TriplePattern<T, T, T>("y", null, "p", null, "x", null));
		engine.addRule(this.r09_symmetric_A);

		this.r10_selfInverse_symmetric.condition().addCondition(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "p", null));
		this.r10_selfInverse_symmetric.action().addAction(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasType, "fix2",
						this.cds.type_symmetricRelation));
		engine.addRule(this.r10_selfInverse_symmetric);

		this.r11_symmetric_selfInverse.condition().addCondition(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasType, "fix2",
						this.cds.type_symmetricRelation));
		this.r11_symmetric_selfInverse.action().addAction(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "p", null));
		engine.addRule(this.r11_symmetric_selfInverse);

		this.r12_inverseCausesSameAs
				.condition()
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "q",
								null))
				.addCondition(
						new TriplePattern<T, T, T>("q", null, "fix1", this.cds.hasInverse, "r",
								null));
		this.r12_inverseCausesSameAs.action().addAction(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.sameAs, "r", null));
		engine.addRule(this.r12_inverseCausesSameAs);
	}

	public void addInverseRules(final IRuleEngine<T, T, T> engine) {
		this.r02_inverseA
				.condition()
				.addCondition(new TriplePattern<T, T, T>("x", null, "p", null, "y", null))
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "q",
								null));
		this.r02_inverseA.action().addAction(
				new TriplePattern<T, T, T>("y", null, "q", null, "x", null));
		engine.addRule(this.r02_inverseA);

		this.r03_inverseT.condition().addCondition(
				new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "q", null));
		this.r03_inverseT.action().addAction(
				new TriplePattern<T, T, T>("q", null, "fix2", this.cds.hasInverse, "p", null));
		// this is done via symmetric now
		// no longer used: engine.addRule(this.r03_inverseT);

		this.r13_inverseType
				.condition()
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasInverse, "q",
								null))
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix2", this.cds.hasType, "t", null));
		this.r13_inverseType.action().patterns()
				.add(new TriplePattern<T, T, T>("q", null, "fix3", this.cds.hasType, "t", null));
		engine.addRule(this.r13_inverseType);
	}

	private double additionalPredicateCosts(final IVariable<T> var) {
		final Object p = var.getExpected();
		if (p == null) {
			return 0;
		}

		// estimate typical frequency, higher frequency = higher costs

		// rare
		if (p.equals(this.cds.hasSubType)) {
			return 0.2;
		}
		if (p.equals(this.cds.hasSuperType)) {
			return 0.2;
		}
		if (p.equals(this.cds.hasInverse)) {
			return 0.2;
		}

		// medium
		if (p.equals(this.cds.sameAs)) {
			return 0.7;
		}
		if (p.equals(this.cds.hasAlias)) {
			return 0.8;
		}
		if (p.equals(this.cds.isAliasOf)) {
			return 0.8;
		}

		// frequent
		if (p.equals(this.cds.hasType)) {
			return 0.9;
		}
		if (p.equals(this.cds.hasInstance)) {
			return 0.9;
		}

		// default
		return 0;
	}

	public void addTransitiveRules(final IRuleEngine<T, T, T> engine) {
		this.r01_transitive
				.condition()

				.addCondition(new TriplePattern<T, T, T>("x", null, "p", null, "y", null))
				.addCondition(new TriplePattern<T, T, T>("y", null, "p", null, "z", null))
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix1", this.cds.hasType, "fix2",
								this.cds.type_transitiveRelation)

				);
		this.r01_transitive.action().addAction(
				new TriplePattern<T, T, T>("x", null, "p", null, "z", null));
		engine.addRule(this.r01_transitive);
	}

	public void addTypeInheritanceRules(final IRuleEngine<T, T, T> engine) {
		this.r04_type_inheritance
				.condition()

				.addCondition(
						new TriplePattern<T, T, T>("x", null, "fix1", this.cds.hasType, "t", null))
				.addCondition(
						new TriplePattern<T, T, T>("t", null, "fix2", this.cds.hasSuperType, "u",
								null));
		this.r04_type_inheritance.action().addAction(
				new TriplePattern<T, T, T>("x", null, "fix3", this.cds.hasType, "u", null));
		engine.addRule(this.r04_type_inheritance);

		this.r05_rel_inheritance
				.condition()

				.addCondition(new TriplePattern<T, T, T>("x", null, "p", null, "y", null))
				.addCondition(
						new TriplePattern<T, T, T>("p", null, "fix2", this.cds.hasSuperType, "u",
								null));
		this.r05_rel_inheritance.action().addAction(
				new TriplePattern<T, T, T>("x", null, "u", null, "y", null));
		engine.addRule(this.r05_rel_inheritance);

		/**
		 * <pre>
		 * ?a		'hasInverse'	?b
		 * ?c		'hasInverse'	?d
		 * ?a		'isSubtypeOf'   ?c
		 *   ==>
		 * ?b       'isSubtypeOf'   ?d
		 * </pre>
		 */
		this.r17_inv_subType_inv
				.condition()

				.addCondition(
						new TriplePattern<T, T, T>("a", null, "fix1", this.cds.hasInverse, "b",
								null))
				.addCondition(
						new TriplePattern<T, T, T>("c", null, "fix2", this.cds.hasInverse, "d",
								null))
				.addCondition(
						new TriplePattern<T, T, T>("a", null, "fix3", this.cds.hasSuperType, "c",
								null));
		this.r17_inv_subType_inv.action().addAction(
				new TriplePattern<T, T, T>("b", null, "fix1", this.cds.hasSuperType, "d", null));
		engine.addRule(this.r17_inv_subType_inv);
	}

	private double estimateCosts(final IVariable<T> var, final ConditionBinding<T, T, T> binding) {
		if (var.isStar() && (binding == null || !binding.isBound(var))) {
			return 1;
		}
		else {
			return 0;
		// IMPROVE deal with fuzzy cases such as !isStar & !isExact
		}
	}

	/**
	 * @param pattern
	 * @return 0 if pattern has no costs (= completely defined).
	 */
	@Override
	public double estimatedCosts(final ITriplePattern<T, T, T> pattern) {
		return estimatedCosts(pattern, null);
	}

	@Override
	public double estimatedCosts(final ITriplePattern<T, T, T> pattern, final ConditionBinding<T, T, T> binding) {
		// simple rating: just count number of open variables
		double open = 0;
		open += estimateCosts(pattern.s(), binding);
		open += estimateCosts(pattern.o(), binding);
		final double pCosts = estimateCosts(pattern.p(), binding);
		open += pCosts;

		if (pCosts == 0 && open > 0) {
			// otherwise: increase cost for more difficult patterns
			open += additionalPredicateCosts(pattern.p());
		}

		return open;
	}

	public static void main(final String[] args) {

		final CdsRules<String> cdsRules = new CdsRules<String>(new CdsIdT.IIdMapper<String>() {

			@Override
			public String toId(final CdsId cdsId) {
				return cdsId.name();
			}
		});
		final RuleEngine<String, String, String> ruleEngine = new RuleEngine<String, String, String>();
		cdsRules.addAxiomaticRulesForCDS(ruleEngine);

		ruleEngine.dumpRules();
	}
}
