package de.xam.p13n.shared;

import java.io.Serializable;

import org.xydra.annotations.RequiresAppEngine;
import org.xydra.annotations.RunsInAppEngine;
import org.xydra.annotations.RunsInGWT;
import org.xydra.annotations.Setting;
import org.xydra.annotations.Template;

import de.xam.p13n.shared.time.MiniTimeZone;

/**
 * A personal choice of timezone (as UTC-offset), and {@link MiniLocale}.
 *
 * Compact string format: 'en_gb_formalZZZ8' where 'en_gb_formal' is the locale
 * and '8' is UTC+8
 *
 * @author xamde
 */
@RunsInGWT(true)
@RunsInAppEngine(true)
@RequiresAppEngine(false)
public class Personalisation implements Serializable {

	private static final long serialVersionUID = 2597415344829634481L;

	/**
	 * en-US,default style, Timezone: San Francisco
	 */
	public static final Personalisation EN_US__SAN_FRANCISCO = new Personalisation(
			MiniLocale.EN_US, MiniTimeZone.US_SAN_FRANCISCO);

	public static final Personalisation GERMANY = new Personalisation(MiniLocale.De_DE,
			MiniTimeZone.GERMANY_BERLIN);

	/**
	 * en-US,default style, Timezone: San Francisco
	 */
	@Setting("Default personalisation for anything: San Francisco, US")
	public static final Personalisation DEFAULT = EN_US__SAN_FRANCISCO;

	public static final String ZZZ = "ZZZ";

	/** for GWT only */
	public Personalisation() {
	}

	public static Personalisation create(final String localeString, final int utcOffset) {
		final MiniLocale locale = SharedLocaleUtils.fromString(localeString);
		return new Personalisation(locale, utcOffset);
	}

	/**
	 * @param compactString
	 *            a compact string obtained via {@link #toCompactString()}
	 * @return a {@link Personalisation} or null (if given string was null)
	 */
	public static Personalisation fromString(final String compactString) {
		if (compactString == null) {
			return null;
		}

		final int zzz = compactString.indexOf(ZZZ);
		if (zzz > 0) {
			// split and parse
			final String langString = compactString.substring(0, zzz);
			final String offsetString = compactString.substring(zzz + ZZZ.length(),
					compactString.length());

			final MiniLocale locale = SharedLocaleUtils.fromString(langString);
			final MiniTimeZone timeZone = MiniTimeZone.fromString(offsetString);
			return new Personalisation(locale, timeZone);
		} else {
			throw new IllegalArgumentException("'" + compactString + "' must contain '" + ZZZ + "'");
		}
	}

	private MiniLocale locale;

	private MiniTimeZone timeZone;

	/**
	 * @param locale
	 *            must not be null
	 * @param utcOffset
	 *            in milliseconds
	 */
	public Personalisation(final MiniLocale locale, final int utcOffset) {
		this.locale = locale;
		this.timeZone = new MiniTimeZone(utcOffset);
	}

	public Personalisation(final MiniLocale locale, final MiniTimeZone timeZone) {
		this.locale = locale;
		this.timeZone = timeZone;
	}

	@Template("MiniLocale")
	public MiniLocale getLocale() {
		return this.locale;
	}

	@Template("MiniTimezone")
	public MiniTimeZone getTimeZone() {
		return this.timeZone;
	}

	@Template("UTC offset")
	public int getUtcOffset() {
		return this.timeZone.getUtcOffset();
	}

	/**
	 * @param precision
	 *            3 for high precision down to 0 = no precision
	 * @return a file extension to denote the given {@link Personalisation}
	 */
	public String toClassifier(final int precision) {
		final String ext = SharedLocaleUtils.toString(this.locale, precision);
		if (ext.length() == 0) {
			return "";
		} else {
			return "_" + ext;
		}
	}

	/**
	 * @return as compact string in the format locale 'ZZZ' utcOffset
	 */
	public String toCompactString() {
		final String c = this.locale + ZZZ + this.timeZone.toCompactString();
		return c;
	}

	@Override
	public String toString() {
		return toCompactString();
	}

	@Override
	public boolean equals(final Object other) {
		return other instanceof Personalisation
				&& ((Personalisation) other).locale.toString().equals(this.locale.toString())
				&& ((Personalisation) other).timeZone.toCompactString().equals(
						this.timeZone.toCompactString());
	}

	@Override
	public int hashCode() {
		return this.locale.hashCode() + this.timeZone.hashCode();
	}
}
