package de.xam.texthtml.text;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.xydra.base.util.DumpUtilsBase;
import org.xydra.index.impl.IntegerRangeIndex;
import org.xydra.index.impl.IntegerRangeIndex.Span;

/**
 * A single string can contain a lot of annotations.
 *
 *
 *
 * @author xamde
 * @param <A> annotation type
 */
public class AnnotatedString<A> {

	/** mark up which character ranges have an annotation */
	IntegerRangeIndex rangeIndex = new IntegerRangeIndex();

	/**
	 * Write down here for each span what the interpretation is. Index annotation at span-start
	 */
	Map<Integer, AnnotatedSpan<A>> annotations = new HashMap<Integer, AnnotatedSpan<A>>();

	public AnnotatedString(final String s) {
		super();
		this.s = s;
	}

	private final String s;

	/**
	 * A kind of {@link Span} with an annotation
	 *
	 * @param <A> annotation type
	 */
	public static class AnnotatedSpan<A> {

		public AnnotatedSpan(final int startInclusive, final int endInclusive, final A annotation) {
			super();
			this.startInclusive = startInclusive;
			this.endInclusive = endInclusive;
			this.annotation = annotation;
		}

		private final int startInclusive;
		private final int endInclusive;

		private final A annotation;

		public A getAnnotation() {
			return this.annotation;
		}

		public int getStartInclusive() {
			return this.startInclusive;
		}

		public int getEndInclusive() {
			return this.endInclusive;
		}

	}

	public String getString() {
		return this.s;
	}

	@Override
	public String toString() {
		final StringBuilder b = new StringBuilder();

		b.append(getString());
		b.append("\n");
		for (int i = 0; i < this.s.length(); i++) {
			b.append(isAnnotated(i) ? 'A' : '-');
		}

		return b.toString();
	}

	public void addAnnotation(final int startInclusive, final int endInclusive, final A annotation) {
		final AnnotatedSpan<A> ann = new AnnotatedSpan<A>(startInclusive, endInclusive, annotation);
		addAnnotation(ann);
	}

	public void addAnnotation(final AnnotatedSpan<A> ann) {
		this.rangeIndex.index(ann.startInclusive, ann.endInclusive);
		this.annotations.put(ann.startInclusive, ann);
	}

	public boolean isAnnotated(final int index) {
		return this.rangeIndex.isInInterval(index);
	}

	/**
	 * @param maxValueInclusive inclusive; used only at then end of the integerRange to append a final span. Typically
	 *        use length() - 1.
	 * @return all spans until maxValue; each span has a boolean flag to indicate, if the span is within an integer
	 *         range or between two of them.
	 */
	public Iterator<Span> spanIterator(final int maxValueInclusive) {
		return this.rangeIndex.spanIterator(maxValueInclusive);
	}

	/**
	 * @return all spans of this {@link AnnotatedString}
	 */
	public Iterator<Span> spanIterator() {
		return this.rangeIndex.spanIterator(0, this.s.length() - 1);
	}

	public Iterator<Span> spanIterator(final int startInvlusive, final int maxValueInclusive) {
		return this.rangeIndex.spanIterator(startInvlusive, maxValueInclusive);
	}

	/**
	 * @param startInclusive
	 * @return
	 */
	public AnnotatedSpan<A> getAnnotationStartingAt(final int startInclusive) {
		return this.annotations.get(startInclusive);
	}

	public void dump() {
		this.rangeIndex.dump();
		DumpUtilsBase.dump(this.annotations);
	}

	/**
	 * @param annSpan
	 * @return @NeverNull
	 */
	public String getString(final AnnotatedSpan<A> annSpan) {
		return this.getString().substring(annSpan.getStartInclusive(), annSpan.getEndInclusive() + 1);
	}

	/**
	 * @param span with startInclusive and endInclusive
	 * @return substring
	 */
	public String getString(final Span span) {
		return this.getString().substring(span.getStartInclusive(), span.getEndInclusive() + 1);
	}

	/**
	 * TODO performance: replace this concept with a resumable iterator which doesnt mind changes to the annotatedString
	 * while it runs (using a simple lastSentSpan shoulde be enough)
	 *
	 * @return a list of all {@link Span} that are not {@link Span#isInRange()}
	 */
	public List<Span> spansNotInRangeList() {
		final List<Span> list = new ArrayList<>();
		final Iterator<Span> spanIt = spanIterator();
		while (spanIt.hasNext()) {
			final Span span = spanIt.next();
			if (span.isInRange()) {
				continue;
			}
			list.add(span);
		}
		return list;
	}

}
