package de.xam.texthtml.text;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Static utilities for handling JspWiki.org syntax.
 *
 * TODO clarify legal syntax of URI vs. IRI
 *
 * @author xamde
 */
public class JspWikiSyntax {

	/** e.g. a 'b' a 'g' or a '-' can be auto-linked. */
	public static final String _AUTOLINKABLE_CHARS_NO_WHITESPACE = "\\w";

	/** everything legal in WikiWords except space */
	public static final String _AUTOLINKABLE_WHITESPACE_CHARS = "\\-_.";

	/** e.g. a 'b' a 'g' or a '-' can be auto-linked. */
	public static final String AUTOLINKABLE_CHARS_WITH_WHITESPACE = _AUTOLINKABLE_CHARS_NO_WHITESPACE
			+ _AUTOLINKABLE_WHITESPACE_CHARS;

	/**
	 * result in group 1, a list of potential wiki words, separated by space. Includes noise such as '-'.
	 */
	public static final Pattern pattern_AUTOLINKABLE_EXPRESSION = Pattern.compile(

	// start of content OR non-autolinkable stuff
			"(?:^|[^" + AUTOLINKABLE_CHARS_WITH_WHITESPACE + "]+)" +

	// group 1
					"(" + "[" + _AUTOLINKABLE_CHARS_NO_WHITESPACE + "]+"

	// note the added space!
					+ "([" + AUTOLINKABLE_CHARS_WITH_WHITESPACE + " ]+"

	+ "[" + _AUTOLINKABLE_CHARS_NO_WHITESPACE + "]+)+"

	+ ")"

	// '?!'=negative lookahead; end of content OR
	// non-autolinkable stuff
					+ "(?:$|(?![" + AUTOLINKABLE_CHARS_WITH_WHITESPACE + "]+))"

	, Pattern.UNICODE_CHARACTER_CLASS);

	public static final Pattern pattern_AUTOLINKABLE_WHITESPACE = Pattern
			.compile("[" + _AUTOLINKABLE_WHITESPACE_CHARS + "]");

	public static final Pattern pattern_EMAIL = Pattern.compile(TextRenderer.EMAIL);

	/**
	 * group 1 is the content of the link.
	 *
	 * can be almost anything: [FooBar], [Foo Bar], [#1], [http://example.com], [Google:This], [Foo|http://example.com]
	 *
	 * Help:
	 *
	 * <pre>
	 * [link]     = create a hyperlink to an internal WikiPage called 'Link'.
	 * [this is also a link] = create a hyperlink to an internal WikiPage called
	 *              'ThisIsAlsoALink' but show the link as typed with spaces.
	 * [a sample|link] = create a hyperlink to an internal WikiPage called
	 *              'Link', but display the text 'a sample' to the
	 *              user instead of 'Link'.
	 * ~NoLink    = disable link creation for the word in CamelCase.
	 * [1]        = make a reference to a footnote numbered 1.
	 * [#1]       = mark the footnote number 1.
	 * [[link]     = create text '[link]'.
	 *
	 * [myUploadedImage.png]
	 * [TextFormattingRules/myUploadedImage.png]
	 *
	 * [wikipagename/attachment.pdf]
	 * [attachment.pdf]
	 * </pre>
	 */
	public static final Pattern pattern_FIND_BRACED_LINK = Pattern.compile(

	"\\["

	+ "(["
	/* almost anything allowed in URLs and email adresses -- except the opening and closing brace */
			+ TextRenderer.LEGAL_URL_CHARS

	// allow path and fragments
			+ "?#"

	// for labeled links
			+ "|"

	// for links to wiki pages
			+ " "

	// for footnotes
	// IMPROVE we ignore footnotes for now
	// +"#"

	// exclude opening and closing brace
			+ "&&[^\\]\\[]"

	+ "]+)"

	+ "\\]"

	);

	public static final Pattern pattern_FIND_EMAIL = Pattern.compile(TextRenderer.FIND_EMAIL);

	public static final Pattern pattern_FIND_FULL_HTTP_URL = Pattern.compile(TextRenderer.FIND_HTTP_URL);

	public static final Pattern pattern_FIND_FILE_URL = Pattern.compile(TextRenderer.FIND_FILE_URL);

	public static final Pattern pattern_FIND_PARTIAL_URL = Pattern.compile(TextRenderer.FIND_PARTIAL_URL);

	public static final Pattern pattern_FOOTNOTE = Pattern.compile("[0-9]{1,3}");

	public static final Pattern pattern_FULL_HTTP_URL = Pattern.compile(TextRenderer.FULL_HTTP_URL);

	public static final Pattern pattern_FILE_URL = Pattern.compile(TextRenderer.FILE_URL);

	public static final Pattern pattern_MINIMAL_URL_CONTAINS = Pattern.compile(TextRenderer.REGEX_MINIMAL_URL_CONTAINS);

	public static final Pattern pattern_PARTIAL_URL = Pattern.compile(TextRenderer.PARTIAL_URL);

	public static final Pattern pattern_WIKILINK = Pattern.compile("\\["

	+ "[^\\[\\]\\n\\t\\r]+"

	+ "\\]", Pattern.UNICODE_CHARACTER_CLASS);

	/** group 1: namespace; group 2: linkName */
	public static final Pattern pattern_WIKILINK_LINKTARGET_NAMESPACE = Pattern.compile(

	"([a-zA-Z]:)"

	+ "([" + TextRenderer.LEGAL_URL_CHARS + "])"

	, Pattern.UNICODE_CHARACTER_CLASS);

	/** includes 'Foo Bar', which needs to be translated to 'FooBar' */
	public static final Pattern pattern_WIKILINK_TARGET = Pattern.compile(

	"[" + TextRenderer.LEGAL_URL_CHARS + " " + "]+"

	);

	/** group 1: label; group 2: linkTarget */
	public static final Pattern pattern_WIKILINK_TARGET_LABELED = Pattern.compile(

	"([" + TextRenderer.LEGAL_URL_CHARS + " " + "&&[^|\\]\\[]" + "]+)"

	+ "[ ]*[|][ ]*"

	+ "([" + TextRenderer.LEGAL_URL_CHARS + " " + "&&[^|\\]\\[]" + "]+)"

	, Pattern.UNICODE_CHARACTER_CLASS);

	public static boolean couldBeEmail(final String s) {
		return s.indexOf('@') > 0;
	}

	public static boolean couldBeURI(final String t) {
		return t.indexOf('.') >= 0 || t.indexOf(':') >= 0 || t.contains("localhost");
	}

	/**
	 * @param line
	 * @return title from a wiki line denoting a headline
	 */
	public static String getHeadlineText(final String line) {
		String s = line;
		while (s.startsWith("!")) {
			s = s.substring(1);
		}
		return s.trim();
	}

	/**
	 * @param t
	 * @return wiki page title from a wiki link "[foo]"
	 */
	public static String getWikiLinkText(final String t) {
		assert isWikiLink(t);
		return t.substring(1, t.length() - 1);
	}

	/**
	 * @param s
	 * @return true iff s is a syntactically valid email address
	 */
	public static boolean isEmailAddress(final String s) {
		return pattern_EMAIL.matcher(s).matches();
	}

	public static boolean isFootnote(final String s) {
		return pattern_FOOTNOTE.matcher(s).matches();
	}

	public static boolean isPluginCall(final String s) {
		return s.startsWith("{INSERT") && s.endsWith("}");
	}

	public static boolean isHeadline(final String line) {
		return line.startsWith("!");
	}

	/**
	 * @param t
	 * @return true if t is an file: link
	 */
	public static boolean isFileLink(final String t) {
		assert t != null;
		return pattern_FILE_URL.matcher(t).matches();
	}

	/**
	 * @param t
	 * @return true if t is an http link
	 */
	public static boolean isHttpLink(final String t) {
		assert t != null;
		return pattern_FULL_HTTP_URL.matcher(t).matches();
	}

	/**
	 * @param s
	 * @return true iff s is a http(s): URI or an email address (without 'mailto:')
	 */
	public static boolean isIri(final String s) {
		assert s != null;
		// IMPROVE allow other IRI syntax, too
		return isHttpLink(s) || isEmailAddress(s) || isFileLink(s);
	}

	public static boolean isWikiLink(final String t) {
		assert t != null;
		return t.startsWith("[") && t.endsWith("]");
	}

	public static boolean isWikiWord(final String t) {
		assert t != null;
		return CamelCase.isCamelCase(t);
	}

	public static void main(final String[] args) {
		System.out.println(pattern_FIND_BRACED_LINK.toString());
		System.out.println(JspWikiSyntax.pattern_AUTOLINKABLE_EXPRESSION);

		final Matcher m = pattern_AUTOLINKABLE_EXPRESSION.matcher("'Blup Foobar'");
		while (m.find()) {
			System.out.println(m.group(1));
		}

	}
}
