package de.xam.texthtml.text;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.xydra.annotations.CanBeNull;
import org.xydra.annotations.LicenseApache;
import org.xydra.annotations.NeverNull;
import org.xydra.annotations.RunsInGWT;
import org.xydra.conf.escape.Escaping;
import org.xydra.index.IIntegerRangeIndex;
import org.xydra.index.impl.DebugUtils;
import org.xydra.index.impl.IntegerRangeIndex;
import org.xydra.log.api.Logger;
import org.xydra.log.api.LoggerFactory;
import org.xydra.sharedutils.URLUtils;

/**
 * see Also {@link Escaping} and {@link EncTool}
 *
 * @author xamde
 */
/**
 * @author xamde
 */
@RunsInGWT(true)
public class TextTool {

	public static final char ESCAPE = '\\';

	private static final Logger log = LoggerFactory.getLogger(TextTool.class);

	/** for debug messages */
	public static final String MARKER_CLOSE = " <__ ";

	/** for debug messages */
	public static final String MARKER_OPEN = " __> ";

	/**
	 * @param codepoint
	 * @param manyCodepoints
	 * @return true if the given codePoint occurs in the given string
	 */
	public static boolean codepointIsOneOf(final int codepoint, final String manyCodepoints) {
		return manyCodepoints.contains("" + (char) codepoint);
	}

	/**
	 * Slow, O(n), but does the job.
	 *
	 * @param set of strings @NeverNull
	 * @param s
	 * @CanBeNull
	 * @param caseMatters
	 * @return true if set contains s
	 */
	public static boolean contains(final Set<String> set, final String s, final boolean caseMatters) {
		assert set != null;
		if (caseMatters) {
			return set.contains(s);
		} else {
			for (final String t : set) {
				if (equals(t, s, false)) {
					return true;
				}
			}
			return false;
		}
	}

	/**
	 * @param s
	 * @NeverNull
	 * @param part
	 * @NeverNull
	 * @param caseMatters
	 * @return true if part is contained in s
	 */
	public static boolean contains(final String s, final String part, final boolean caseMatters) {
		assert s != null;
		assert part != null;
		if (caseMatters) {
			return s.contains(part);
		} else {
			return s.toLowerCase().contains(part.toLowerCase());
		}
	}

	/**
	 * Not unicode safe (outside BMP).
	 *
	 * @param string
	 * @CanBeNull
	 * @param chars
	 * @NeverNull
	 * @return true if the string contains at least one of the given 'chars' anywhere
	 */
	public static boolean containsOneOf(final String string, final char[] chars) {
		if (string == null || string.length() == 0) {
			return false;
		}
		if (chars == null) {
			throw new IllegalArgumentException("char array null");
		}

		for (int s = 0; s < string.length(); s++) {
			final char s_char = string.charAt(s);
			if (isOneOf(s_char, chars)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Unicode safe.
	 *
	 * @param string
	 * @CanBeNull
	 * @param chars
	 * @NeverNull
	 * @return true if the string contains at least one of the given 'chars' anywhere
	 */
	public static boolean containsOneOf(final String string, final IntegerRangeIndex chars) {
		if (string == null || string.length() == 0) {
			return false;
		}
		if (chars == null) {
			throw new IllegalArgumentException("chars is null");
		}

		int i = 0;
		while (i < string.length()) {
			final int c = string.codePointAt(i);

			if (isOneOf(c, chars)) {
				return true;
			}

			i += Character.charCount(c);
		}

		return false;
	}

	/**
	 * Not unicode safe (outside BMP).
	 *
	 * @param string
	 * @CanBeNull
	 * @param chars
	 * @NeverNull
	 * @return true if the string contains only characters listed in chars or if the string is null or empty
	 */
	public static boolean containsOnly(final String string, final char[] chars) {
		if (string == null || string.length() == 0) {
			return true;
		}
		if (chars == null || chars.length == 0) {
			throw new IllegalArgumentException("char array empty");
		}

		for (int s = 0; s < string.length(); s++) {
			final char s_char = string.charAt(s);
			if (!isOneOf(s_char, chars)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Unicode safe.
	 *
	 * @param string @CanBeNull
	 * @param legal @NeverNull
	 * @return true if the string contains only characters listed in legal or if the string is null or empty
	 */
	public static boolean containsOnly(final String string, final IIntegerRangeIndex legal) {
		if (string == null || string.length() == 0) {
			return true;
		}
		if (legal == null) {
			throw new IllegalArgumentException("char array empty");
		}

		int i = 0;
		while (i < string.length()) {
			final int codePoint = string.codePointAt(i);
			if (!legal.isInInterval(codePoint)) {
				return false;
			}
			i += Character.charCount(codePoint);
		}
		return true;
	}

	/**
	 * @param regex
	 * @param s
	 * @return the number of consecutive, non-overlapping matches of regex starting at the begin of the string
	 */
	public static int countConsecutiveMatches(final String regex, final String s) {
		String t = s;
		final Pattern p = Pattern.compile("^(" + regex + ").*");
		int c = 0;
		Matcher m = p.matcher(t);
		while (m.matches()) {
			log.trace("match = '" + m.group(1) + "'");
			final int l = m.group(1).length();
			c += 1;
			t = t.substring(l);
			m = p.matcher(t);
		}

		return c;
	}

	/**
	 * @param marker
	 * @param s
	 * @return number of times that marker appears in s, starting from 0, not interrupted by other characters
	 */
	public static int countNumberOfMarkerRepetitionsFromBeginOfString(final String marker, final String s) {
		int count = 0;
		int i = 0;
		while (i >= 0) {
			i = s.indexOf(marker, i);
			if (i >= 0) {
				count++;
				i += marker.length();
			} else {
				break;
			}
		}
		return count;
	}

	/**
	 * @param a
	 * @NeverNull
	 * @param b
	 * @NeverNull
	 * @param taboo these are ignored (skipped over) when comparing strings a and b
	 * @param caseMatters if true, characters are compared with case
	 * @return true iff a and b contain the same (casing...) characters (ignoring the taboo characters) in the same
	 *         order
	 */
	public static boolean equalIgnoringTabooCharacters(final String a, final String b, final IIntegerRangeIndex taboo,
			final boolean caseMatters) {
		assert a != null;
		assert b != null;
		int aIndex = 0;
		int bIndex = 0;
		int aCodePoint = -1;
		int bCodePoint = -1;

		// advance until at least one string ended
		while (aIndex < a.length() && bIndex < b.length()) {

			// advance in a until we are done or have a non-taboo character
			while (aIndex < a.length()) {
				aCodePoint = a.codePointAt(aIndex);
				aIndex += Character.charCount(aCodePoint);
				if (isNoneOf(aCodePoint, taboo)) {
					break;
				}
			}
			// advance in b until we are done or have a non-taboo character
			while (bIndex < b.length()) {
				bCodePoint = b.codePointAt(bIndex);
				bIndex += Character.charCount(bCodePoint);
				if (isNoneOf(bCodePoint, taboo)) {
					break;
				}
			}

			// compare single, non-taboo codePoints
			if (!equals(aCodePoint, bCodePoint, caseMatters)) {
				return false;
			}
		}
		// maybe a or b or both ended

		// check if there is only trailing taboo in a
		while (aIndex < a.length()) {
			aCodePoint = a.codePointAt(aIndex);
			aIndex += Character.charCount(aCodePoint);
			if (isNoneOf(aCodePoint, taboo)) {
				return false;
			}
		}
		// check if there is only trailing taboo in b
		while (bIndex < b.length()) {
			bCodePoint = b.codePointAt(bIndex);
			bIndex += Character.charCount(bCodePoint);
			if (isNoneOf(bCodePoint, taboo)) {
				return false;
			}
		}

		return true;
	}

	/**
	 * @param a
	 * @NeverNull
	 * @param b
	 * @NeverNull
	 * @param caseMatters if true, characters are compared with case
	 * @return true iff a and b contain the same alphanumerics (regex: all except punctuation) in the same order
	 */
	public static boolean equalNonWhitespaceCharacters(final String a, final String b, final boolean caseMatters) {
		assert a != null;
		assert b != null;
		return equalIgnoringTabooCharacters(a, b, Unicodes.unicodePureSeparator, caseMatters);
	}

	/**
	 * @param codePointA
	 * @param codePointB
	 * @param caseMatters if false, codePoints are compared in their lowerCased forms
	 * @return true if the two codePoints are the same - see caseMatters
	 */
	public static boolean equals(final int codePointA, final int codePointB, final boolean caseMatters) {
		if (caseMatters) {
			return codePointA == codePointB;
		} else {
			return Character.toLowerCase(codePointA) == Character.toLowerCase(codePointB);
		}
	}

	/**
	 * @param a
	 * @CanBeNull
	 * @param b
	 * @CanBeNull
	 * @param caseMatters iff true, casing of a and b is respected
	 * @return true iff a==b==null or if a and b have the same content
	 */
	public static boolean equals(final String a, final String b, final boolean caseMatters) {

		if (a == null) {
			return b == null;
		}
		if (b == null) {
			return false;
		}

		if (caseMatters) {
			return a.equals(b);
		} else {
			return a.equalsIgnoreCase(b);
		}
	}

	/**
	 * @param s
	 * @CanBeNull
	 * @return null if input is null @CanBeNull
	 */
	public static String firstLetterUppercase(final String s) {
		if (s == null) {
			return null;
		}
		if (s.length() == 1) {
			return s.toUpperCase();
		}
		return s.substring(0, 1).toUpperCase() + s.substring(1, s.length());
	}

	/**
	 * Replace real line break with '|\\n|' inline
	 *
	 * @param s
	 * @return
	 */
	public static String foldLineBreaks(final String s) {
		return s.replace("\n", "|\\n|");
	}

	/**
	 * @param s a string
	 * @param pos in s
	 * @param before how many chars before pos
	 * @param after how many chars after pos
	 * @return a shorter string using ASCII art to highlight a position in a string
	 */
	public static String highlightPos(final String s, final int pos, final int before, final int after) {
		if (s == null) {
			return "(empty string)";
		}

		if (pos == -1) {
			// ignore pos
			final int maxLen = before + 1 + after;
			return s.substring(0, Math.min(s.length(), maxLen));
		}

		if (s.length() < pos) {
			throw new IllegalArgumentException("String len " + s.length() + " <= pos " + pos);
		}

		final int start = Math.max(pos - before, 0);
		final int end = Math.min(pos + 1 + after, s.length());

		final String pre = s.substring(start, pos);
		if (s.length() == pos) {
			return pre + MARKER_CLOSE + "END";
		}

		final String highlight = s.substring(pos, pos + 1);
		final String post = s.substring(pos + 1, end);
		return pre + MARKER_OPEN + highlight + MARKER_CLOSE + post;
	}

	public static String indent(final int count, final String indent) {
		final StringBuilder buf = new StringBuilder();
		for (int i = 0; i < count; i++) {
			buf.append(indent);
		}
		return buf.toString();
	}

	/**
	 * @param codePoint
	 * @param taboo
	 * @NeverNull
	 * @return true if the codePoint is none of the taboo characters
	 */
	public static boolean isNoneOf(final int codePoint, final char[] taboo) {
		assert taboo != null;
		for (int i = 0; i < taboo.length; i++) {
			if (codePoint == taboo[i]) {
				return false;
			}
		}
		return true;
	}

	/**
	 * @param codePoint
	 * @param taboo
	 * @NeverNull
	 * @return true if the codePoint is none of the taboo characters
	 */
	public static boolean isNoneOf(final int codePoint, final IIntegerRangeIndex taboo) {
		assert taboo != null;
		return !taboo.isInInterval(codePoint);
	}

	/**
	 * @param codePoint
	 * @param legal @NeverNull
	 * @return true if the codePoint is one of the legal chars
	 */
	public static boolean isOneOf(final int codePoint, final char[] legal) {
		assert legal != null;
		for (int i = 0; i < legal.length; i++) {
			if (codePoint == legal[i]) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @param codePoint
	 * @param legal
	 * @NeverNull
	 * @return true if the codePoint is one of the legal chars
	 */
	public static boolean isOneOf(final int codePoint, final IIntegerRangeIndex legal) {
		assert legal != null;
		return legal.isInInterval(codePoint);
	}

	/**
	 * @param codePoint
	 * @param legal
	 * @NeverNull
	 * @return true if the codePoint is one of the legal chars
	 */
	public static boolean isOneOf(final int codePoint, final int[] legal) {
		assert legal != null;
		for (int i = 0; i < legal.length; i++) {
			if (codePoint == legal[i]) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @param dirty
	 * @param taboo @CanBeNull
	 * @return
	 */
	public static String removeAllOf(final String dirty, final int... taboo) {
		if (taboo == null || taboo.length == 0) {
			return dirty;
		}

		final StringBuilder b = new StringBuilder();
		int i = 0;
		while (i < dirty.length()) {
			final int c = dirty.codePointAt(i);

			if (isOneOf(c, taboo)) {
				// skip it
			} else {
				// take it
				b.appendCodePoint(c);
			}

			i += Character.charCount(c);
		}

		return b.toString();
	}

	/**
	 * Removes all taboo characters. If inverted, keeps only taboo characters.
	 *
	 * @param dirty
	 * @param taboo @CanBeNull
	 * @param invert default is false
	 * @return
	 */
	public static String removeAllOf(final String dirty, final IIntegerRangeIndex taboo, final boolean invert) {
		if (taboo == null || taboo.isEmpty()) {
			return dirty;
		}

		final StringBuilder b = new StringBuilder();
		int i = 0;

		while (i < dirty.length()) {
			final int c = dirty.codePointAt(i);

			if (invert) {
				if (isOneOf(c, taboo)) {
					// take it
					b.appendCodePoint(c);
				} else {
					// skip it
				}
			} else {
				if (isOneOf(c, taboo)) {
					// skip it
				} else {
					// take it
					b.appendCodePoint(c);
				}

			}
			i += Character.charCount(c);
		}

		return b.toString();
	}

	@NeverNull
	public static StringBuilder join(final Collection<String> coll, final char sep, final boolean escapeComponents) {
		if (coll.isEmpty()) {
			return new StringBuilder();
		}

		final StringBuilder buf = new StringBuilder();
		final Iterator<String> it = coll.iterator();
		String element = it.next();
		buf.append(escapeComponents ? escape(element) : element);
		while (it.hasNext()) {
			buf.appendCodePoint(sep);
			element = it.next();
			buf.append(escapeComponents ? escape(element) : element);
		}
		return buf;
	}

	@NeverNull
	public static StringBuilder joinStringBuilder(final Collection<StringBuilder> coll, final char sep) {
		if (coll.isEmpty()) {
			return new StringBuilder();
		}

		final StringBuilder buf = new StringBuilder();
		final Iterator<StringBuilder> it = coll.iterator();
		buf.append(it.next());
		while (it.hasNext()) {
			buf.appendCodePoint(sep);
			buf.append(it.next());
		}
		return buf;
	}

	/**
	 * @param raw a string which may contain '\\', '\n', '\t', '\ r', or Unicode '\ uXXXX' where XXXX is hex. The space
	 *        is not there.
	 * @return a string in which java and unicode escapes have been replaced with the correct unicode codepoint
	 */
	public static String materializeEscapes(final String raw) {
		assert raw != null;
		return Escaping.materializeEscapes(raw, false, false);
	}

	/**
	 * Encodes '\' to [backslash][backslash], newline to [backslash][newline], tabs to [backslash][tab], unicode
	 * characters beyond 255 into their '\ uXXXX' codes.
	 *
	 * @param raw
	 * @CanBeNull
	 * @return @CanBeNull if input is null
	 */
	public static String escape(final String raw) {
		if (raw == null) {
			return null;
		}
		if (raw.length() == 0) {
			return "";
		}

		return Escaping.escape(raw, false, true);
	}

	/**
	 * IMPROVE add many more unicode variants?
	 *
	 * @param raw
	 * @return replace all CR-LF-pairs and all single CRs with LFs (\n)
	 */
	public static String normalizeNewlines(final String raw) {
		String s = raw;
		s = s.replace("\r\n", "\n");
		s = s.replace("\r", "\n");
		return s;
	}

	/**
	 * Format a string s so that it has the desired length. Do this by prefixing the string with copies of a padding
	 * string.
	 *
	 * TODO weird behavior depending on length of padding string
	 *
	 * @param padding
	 * @param s
	 * @param desiredTotalLength
	 * @return
	 */
	public static String padLeft(final String padding, final String s, final int desiredTotalLength) {
		if (s.length() >= desiredTotalLength) {
			return s;
		}
		final int missing = desiredTotalLength - s.length();
		final int adding = missing / padding.length();
		final StringBuilder buf = new StringBuilder();
		for (int i = 0; i < adding; i++) {
			buf.append(padding);
		}
		buf.append(s);
		return buf.toString();
	}

	/**
	 * @param padding
	 * @param s
	 * @param desiredTotalLength (minimum)
	 * @return
	 */
	public static String padRight(final String padding, final String s, final int desiredTotalLength) {
		if (s.length() >= desiredTotalLength) {
			return s;
		}
		final int missing = desiredTotalLength - s.length();
		final int adding = missing / padding.length();
		final StringBuilder buf = new StringBuilder();
		buf.append(s);
		for (int i = 0; i < adding; i++) {
			buf.append(padding);
		}
		return buf.toString();
	}

	/**
	 * Split s but respect strings enclosed in single or double quotes. Respects escaped quotes using backslash. Escape
	 * sequences remain always unchanged. Whitespace between tokens is removed (space, tab, CR, LF). Empty strings are
	 * not reported.
	 *
	 * @param s
	 * @param separators e.g. ',' or '|' or ';' or all of them ',;|'. Reserved separators are (') and (") if stripQuotes
	 *        is active.
	 * @param stripQuotes if true, removes the quotes
	 * @return individual tokens. String quote characters have been removed.
	 */
	public static String[] split(final String s, final String separators, final boolean stripQuotes) {
		// assert !separators.contains("\n");
		// assert !separators.contains("\r");
		// assert !separators.contains(" ");
		// assert !separators.contains("\t");
		assert!separators.contains("\'") : "reserved for string-delimiting";
		assert!separators.contains("\"") : "reserved for string-delimiting";
		assert!separators.contains("\\") : "reserved for escape-processing";
		final List<String> tokenList = new ArrayList<>();
		StringBuilder currentToken = new StringBuilder();
		int lastQuote = -1;

		int index = 0;
		while (index < s.length()) {
			final int codepoint = s.codePointAt(index);
			if (codepoint == ESCAPE) {
				currentToken.appendCodePoint(codepoint);
				if (index + 1 < s.length()) {
					currentToken.appendCodePoint(s.codePointAt(index + 1));
					index++;
				}
			} else if (lastQuote == -1) {
				// outside of quoted string

				// IMPROVE not Unicode safe outside of BMP
				if (codepointIsOneOf(codepoint, separators)) {
					// a token finished
					if (currentToken.length() > 0) {
						tokenList.add(trim(currentToken.toString()));
					}
					currentToken = new StringBuilder();
					lastQuote = -1;
				} else if (codepoint == '\'' || codepoint == '\"') {
					lastQuote = codepoint;
					if (!stripQuotes) {
						currentToken.appendCodePoint(codepoint);
					}
				} else {
					currentToken.appendCodePoint(codepoint);
				}
			} else {
				// inside a quoted string
				if (codepoint == lastQuote) {
					// a string finished
					if (!stripQuotes) {
						currentToken.appendCodePoint(codepoint);
					}
					lastQuote = -1;
				} else {
					currentToken.appendCodePoint(codepoint);
				}
			}
			index += Character.charCount(codepoint);
		}
		// add last token
		if (currentToken.length() > 0) {
			tokenList.add(trim(currentToken.toString()));
		}

		return tokenList.toArray(new String[tokenList.size()]);
	}

	/**
	 * @param s
	 * @NeverNull
	 * @param prefix
	 * @NeverNull
	 * @param caseMatters if false, lowerCase is used
	 * @return true if s starts with prefix - see caseMatters
	 */
	public static boolean startsWith(final String s, final String prefix, final boolean caseMatters) {
		assert s != null;
		assert prefix != null;
		if (caseMatters) {
			return s.startsWith(prefix);
		} else {
			return s.toLowerCase().startsWith(prefix.toLowerCase());
		}
	}

	/**
	 * @param str
	 * @NeverNull
	 * @param start inclusive;
	 * @param end exclusive; use -1 for str.length;
	 * @return a substring from start to end
	 */
	public static String substring(final String str, final int start, final int end) {
		assert str != null;
		assert start >= 0;
		assert start <= str.length();
		assert end == -1 || end >= 0 && end <= str.length();

		if (end == -1) {
			return str.substring(start);
		} else {
			return str.substring(start, end);
		}
	}

	/**
	 * @param object
	 * @CanBeNull or empty
	 * @param maxLen
	 * @return a String with max len 'maxLen'
	 */
	public static String toLimitedString(final Object object, final int maxLen) {
		return DebugUtils.toLimitedString(object, maxLen);
	}

	/**
	 * Unicode-safe.
	 *
	 * Think of this as "upper-case union" or as "case-OR".
	 *
	 * @param a must be equalsIgnoreCase the same as b @NeverNull
	 * @param b must be equalsIgnoreCase the same as a @NeverNull
	 * @return a string in which each character which is upper-case in a or b is upper-case.
	 */
	public static String toMergedUpperCase(final String a, final String b) {
		if (a == null) {
			throw new IllegalArgumentException();
		}
		if (b == null) {
			throw new IllegalArgumentException();
		}
		if (a.length() != b.length()) {
			throw new IllegalArgumentException("a='" + a + "' b='" + b + "'");
		}
		if (!a.equalsIgnoreCase(b)) {
			throw new IllegalArgumentException("a='" + a + "' b='" + b + "'");
		}

		final StringBuilder res = new StringBuilder();
		int r = 0;
		while (r < a.length()) {
			final int ac = a.codePointAt(r);
			if (Character.isUpperCase(ac)) {
				res.appendCodePoint(ac);
			} else {
				res.appendCodePoint(b.codePointAt(r));
			}
			r += Character.charCount(ac);
		}
		return res.toString();
	}

	/**
	 * @param s
	 * @return trim ALL whitepace at begin and end, including nbsp;
	 */
	public static String trim(final String s) {
		assert s != null;
		if (s.length() == 0) {
			return s;
		}

		int i = 0;
		// trim at start: search first non-whitespace
		boolean searching = true;
		while (i < s.length() && searching) {
			final int c = s.codePointAt(i);
			if (isWhitespace(c)) {
				// move on
				i += Character.charCount(i);
			} else {
				searching = false;
			}
		}

		if (i == s.length()) {
			return "";
		}

		/* inclusive */
		final int startInclusive = i;
		/* now look backwards from end of string, this only works because our whitespace is single-char unicode (BMP) */
		i = s.length() - 1;
		searching = true;
		while (i >= startInclusive && searching) {
			final int c = s.codePointAt(i);

			// is it a valid UTF-16 codepoint?
			if (!Character.isValidCodePoint(c)) {
				// we need to backtrack one more
				i--;
			}

			if (isWhitespace(c)) {
				// fine, move on, backwards
				i -= 1;
			} else {
				searching = false;
			}
		}
		// is it a valid UTF-16 codepoint?
		if (!Character.isValidCodePoint(s.codePointAt(i))) {
			// we need to backtrack one more
			i--;
		}

		final int endInclusive = i;

		assert startInclusive <= endInclusive : "failed for '" + s + "' start=" + startInclusive + " end="
				+ endInclusive;
		return s.substring(startInclusive, endInclusive + 1);
	}

	/**
	 * @param c
	 * @return true iff [ ][\n][\t][\r][&nbsp;]
	 */
	public static boolean isWhitespace(final int c) {
		switch (c) {
		case ' ':
		case '\t':
		case '\r':
		case '\n':
			// &nbsp;
		case '\u00A0':
			// continue
			return true;
		default:
			return false;
		}
	}

	/**
	 * Just calling {@link URLUtils}
	 *
	 * @param raw
	 * @return
	 */
	public static String urlEncode(final String raw) {
		return URLUtils.encode(raw);
	}

	/**
	 * Just calling {@link URLUtils}
	 *
	 * @param enc
	 * @return
	 */
	public static String urlDecode(final String enc) {
		return URLUtils.decode(enc);
	}

	// public static final String REGEX_VAR = "(?s).*(\\$\\{[^}]+\\}).*";
	public static final String REGEX_VAR =
	// start
	"(?s).*("
			// ${varname}
			+ "(?:\\$\\{([^}]+)\\})" +
			// or
			"|"
			// xxx-varname-xxx, here varname may not contain a '-'
			+ "(?:xxx-([^-]+)-xxx)"
			// end
			+ ").*";

	public static final Pattern REGEX_VAR_PATTERN = Pattern.compile(REGEX_VAR);

	/**
	 * A fork lives in DataMapStoreTools
	 *
	 * @param raw
	 * @param simpleMap
	 * @return interpolated string
	 */
	public static String interpolate(final String raw, final Map<String, String> simpleMap) {
		return interpolate(raw, new IReplacements() {

			@Override
			public String getReplacement(final String varName) {
				return simpleMap.get(varName);
			}
		});
	}

	public static interface IReplacements {
		/**
		 * @param varName
		 * @return null if varName has no defined replacement
		 */
		String getReplacement(String varName);
	}

	/**
	 * Replaces all ${varname} matches with replacements
	 *
	 * @param raw
	 * @param replacements
	 * @return the interpolated string
	 */
	public static String interpolate(final String raw, final IReplacements replacements) {
		if (raw == null) {
			return null;
		}

		String done = raw;
		Matcher m = REGEX_VAR_PATTERN.matcher(done);
		int replacedVariables = 0;
		while (m.matches()) {
			final String varString = m.group(1);
			final String varName = m.group(2) == null ? m.group(3) : m.group(2);
			final String replacement = replacements.getReplacement(varName);
			if (replacement == null) {
				log.warn("Could not interpolate " + varString);
				break;
			} else {
				log.trace("Replacing '" + varName + "' with '" + replacement + "'");
				done = done.replace(varString, replacement);
				m = REGEX_VAR_PATTERN.matcher(done);
				replacedVariables++;
			}
		}
		log.debug("Replaced " + replacedVariables);
		return done;
	}

	/**
	 * @param s
	 * @param separator
	 * @return the last token resulting a tokenisation with 'separator'
	 */
	public static String lastPartSeparatedBy(final String s, final String separator) {
		final int i = s.lastIndexOf(separator);
		if (i < 0 || i + separator.length() > s.length()) {
			return null;
		}
		return s.substring(i + separator.length());
	}

	/**
	 * Overly long words (e.g. URLs) are not broken up.
	 *
	 * @param s
	 * @param maxLineLenght
	 * @return
	 */
	public static String wrap(final String s, final int maxLineLenght) {
		return wrap(s, maxLineLenght, "\n", false);
	}

	/**
	 * <p>
	 * Wraps a single line of text, identifying words by <code>' '</code>.
	 * </p>
	 *
	 * <p>
	 * Leading spaces on a new line are stripped. Trailing spaces are not stripped.
	 * </p>
	 *
	 * <pre>
	 * WordUtils.wrap(null, *, *, *) = null
	 * WordUtils.wrap("", *, *, *) = ""
	 * </pre>
	 *
	 * @param str the String to be word wrapped, may be null
	 * @param wrapLength the column to wrap the words at, must be at least 1
	 * @param newLineStr the string to insert for a new line
	 * @param wrapLongWords true if long words (such as URLs) should be wrapped
	 * @return a line with newlines inserted, <code>null</code> if null input
	 */
	@LicenseApache(project = "Apache Commons Lang", copyright = "Copyright 2001-2011 The Apache Software Foundation")
	public static String wrap(final String str, final int wrapLength, final String newLineStr,
			final boolean wrapLongWords) {
		if (str == null) {
			return null;
		}

		assert newLineStr != null;
		assert wrapLength >= 1;
		final int inputLineLength = str.length();
		int offset = 0;
		final StringBuilder wrappedLine = new StringBuilder(inputLineLength + 32);

		while (inputLineLength - offset > wrapLength) {
			if (str.charAt(offset) == ' ') {
				offset++;
				continue;
			}
			int spaceToWrapAt = str.lastIndexOf(' ', wrapLength + offset);

			if (spaceToWrapAt >= offset) {
				// normal case
				wrappedLine.append(str.substring(offset, spaceToWrapAt));
				wrappedLine.append(newLineStr);
				offset = spaceToWrapAt + 1;

			} else {
				// really long word or URL
				if (wrapLongWords) {
					// wrap really long word one line at a time
					wrappedLine.append(str.substring(offset, wrapLength + offset));
					wrappedLine.append(newLineStr);
					offset += wrapLength;
				} else {
					// do not wrap really long word, just extend beyond limit
					spaceToWrapAt = str.indexOf(' ', wrapLength + offset);
					if (spaceToWrapAt >= 0) {
						wrappedLine.append(str.substring(offset, spaceToWrapAt));
						wrappedLine.append(newLineStr);
						offset = spaceToWrapAt + 1;
					} else {
						wrappedLine.append(str.substring(offset));
						offset = inputLineLength;
					}
				}
			}
		}

		// Whatever is left in line is short enough to just pass through
		wrappedLine.append(str.substring(offset));

		return wrappedLine.toString();
	}

	/**
	 * Simplistic title case, only first char is processed
	 *
	 * @param s
	 * @return 'Aaaaaa aaaaa'
	 */
	public static String toTitleCase(final String s) {
		if (s == null) {
			return null;
		}

		if (s.length() == 0) {
			return s;
		}

		if (s.length() == 1) {
			return s.toUpperCase();
		}

		return s.substring(0, 1).toUpperCase() + s.substring(1).toLowerCase();
	}

	/**
	 * @param a
	 * @param b
	 * @param caseMatters even if case does not matter, it is used to induce a stable sort order on otherwise equal
	 *        strings. Uppercase comes first.
	 * @return comparing strings
	 */
	public static int compare(final String a, final String b, final boolean caseMatters) {
		if (caseMatters) {
			return a.compareTo(b);
		}
		final int c = a.toLowerCase().compareTo(b.toLowerCase());
		if(c==0) {
			// sort by casing if same when ignoring case
			return a.compareTo(b);
		}
		return c;
	}

	/**
	 * Sort in case-sensitive/insensitive way
	 *
	 * @param stringList
	 * @param caseMatters
	 */
	public static void sort(final List<String> stringList, final boolean caseMatters) {
		Collections.sort(stringList, new Comparator<String>() {

			@Override
			public int compare(final String o1, final String o2) {
				return TextTool.compare(o1, o2, caseMatters);
			}
		});
	}

	/**
	 * Count number of characters in given rangeIndex starting from start.
	 *
	 * Unicode-safe.
	 *
	 * @param source
	 * @param start
	 * @param endExclusive
	 * @param legal
	 * @return number of chars (can be more than codepoints) which where all in rangeIndex
	 */
	public static int countLegalCharacters(final String source, final int start, final int endExclusive,
			final IIntegerRangeIndex legal) {
		int count = 0;
		int i = start;
		while (i < endExclusive) {
			final int c = source.codePointAt(i);
			if (legal.isInInterval(c)) {
				count += Character.charCount(c);
			}
			i += Character.charCount(c);
		}
		return count;
	}

	public static boolean equals(final Set<String> a, final Set<String> b, final boolean caseMatters) {
		if (!caseMatters) {
			return a.equals(b);
		}

		if (a.size() != b.size()) {
			return false;
		}

		final Set<String> aLowercase = new HashSet<String>(a.size());
		for (final String s : a) {
			aLowercase.add(s.toLowerCase());
		}

		for (final String bString : b) {
			if (!aLowercase.contains(bString.toLowerCase())) {
				return false;
			}
		}

		return true;
	}

	/**
	 * @param s @CanBeNull
	 * @return s lower-cased @CanBeNull if input is null
	 */
	public static String lowercase(@CanBeNull final String s) {
		if (s == null) {
			return null;
		}
		if (s.length() == 0) {
			return s;
		}
		return s.toLowerCase();
	}

	public static String removeIllegalOrDiscouragedXml10Characters(final String contentString) {
		return TextTool.removeAllOf(contentString, Unicodes.legalEncouragedXml10, true);
	}

	/**
	 * @param s
	 * @return true iff s contains only upper-case characters
	 */
	public static boolean isUpperCase(final String s) {
		return containsOnly(s, Unicodes.unicodeUpperCase);
	}

	/**
	 * @param s
	 * @return true iff s contains only lower-case characters
	 */
	public static boolean isLowerCase(final String s) {
		return containsOnly(s, Unicodes.unicodeLowerCase);
	}

	/**
	 * @param email address
	 * @return email with normalized domain name (part before is NOT lower-cased, as the RFC for email addresses
	 *         mandates)
	 */
	public static String toLowercasedEmail(final String email) {
		final int i = email.indexOf('@');
		assert i > 0 : "found not @-sign in email address '" + email + "'";
		final String localPart = email.substring(0, i);
		final String hostPart = email.substring(i + 1);
		return localPart + "@" + hostPart.toLowerCase();
	}

	/**
	 * @param uri
	 * @return host part lowercased
	 */
	public static String toLowercasedURI(final URI uri) {
		String uriString = uri.toString();
		final String host = uri.getHost();
		if (host == null) {
			// FIXME happens often, e.g. for JspWiki namespaced links
			log.warn("A valid URL without host: '" + uri + "'");
		} else {
			final int i = uriString.indexOf(host);
			uriString = uriString.substring(0, i) + host.toLowerCase() + uriString.substring(i + host.length());
		}
		return uriString;
	}

	public static Comparator<String> createComparator(final boolean caseMatters) {
		return new Comparator<String>() {

			@Override
			public int compare(final String a, final String b) {
				return TextTool.compare(a, b, caseMatters);
			}};
	}
}
